# coding: utf-8
import json
import os
from datetime import datetime

import requests
import xmltodict

import pandas as pd


class YcClient:

    @classmethod
    def __init__(cls, app_key, app_token):
        cls.app_key = app_key
        cls.app_token = app_token
        cls.base_url = "http://8.210.223.221/default/svc/web-service"

    @classmethod
    def call_service(cls, service, params_json):
        """
            调用 SOAP 服务的封装函数。
            :param params_json: 请求的数据内容，Python 字典
            :param service: 要调用的接口方法
            :return: 响应内容
        """
        # 构造 SOAP 请求的 XML 数据
        payload = f"""<?xml version="1.0" encoding="UTF-8"?>
            <SOAP-ENV:Envelope xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/" xmlns:ns1="http://www.example.org/Ec/">
                <SOAP-ENV:Body>
                    <ns1:callService>
                        <appToken>{cls.app_token}</appToken>
                        <appKey>{cls.app_key}</appKey>
                        <service>{service}</service>
                        <paramsJson>{json.dumps(params_json)}</paramsJson>
                    </ns1:callService>
                </SOAP-ENV:Body>
            </SOAP-ENV:Envelope>"""

        # 发送请求
        response = requests.post(cls.base_url, data=payload, headers={"Content-Type": "text/xml;charset=UTF-8"})
        # 检查响应状态码
        if response.status_code != 200:
            raise Exception(f"HTTP 错误: {response.status_code}, 内容: {response.text}")

        # 将 XML 转换为字典
        response_dict = xmltodict.parse(response.text)
        resp = (response_dict.get("SOAP-ENV:Envelope", {}).
                get("SOAP-ENV:Body", {}).
                get("ns1:callServiceResponse", {}).
                get('response', {}))

        json_res = resp
        # 如果 resp 是字符串，则解析为 JSON 字典
        if isinstance(resp, str):
            json_res = json.loads(resp)

        if json_res.get("ask", "Failure") != "Success":
            raise Exception(json_res.get("message", "未知错误"))

        return json_res.get("data", {})

    @classmethod
    def get_products(cls, sku: str) -> dict:
        """
            获取商品列表
            :param sku: 商品编码
        """
        params_json = {
            "pageSize": 10,
            "page": 1,
            "product_sku": sku,
        }

        return cls.call_service("getProductList", params_json)

    @classmethod
    def create_asn(cls, warehouse_code, desc, tracking_number, order_item_list):
        """创建入库单"""
        params_json = {
            "reference_no": tracking_number,  # 参考号
            "warehouse_code": warehouse_code,  # 目的仓
            "tracking_number": tracking_number,  # 跟踪号
            "bulk_cargo_type": 1,  # 散货类型（托）
            "pallet_cnt": 1,  # 散货托数量
            "receiving_desc": desc,  # 描述
            "items": order_item_list
        }

        return cls.call_service("createAsn", params_json)


class Process(YcClient):
    def __init__(self):
        # 调用 API
        key = "c906cc46bda8cea593c0b6b20eadb1f2"
        token = "2832175c4ee5c6efd79e129e919f2dfd"
        super().__init__(key, token)

    @classmethod
    def read_data(cls, file):
        # 读取 Excel 文件
        result = pd.read_excel(file)

        # 按入库单跟踪号和 SKU 去重并汇总数量
        result = result.groupby(['Shipment Request ID', 'SKU'], as_index=False).agg({'Return quantity': 'sum'})

        # 组装结果数据
        result_list = {}
        for _, item in result.iterrows():
            tracking_number = item.get('Shipment Request ID', '')
            order_item = {
                "product_sku": item.get('SKU', ''),
                "quantity": item.get('Return quantity', 0),
                "box_no": 1
            }

            # 初始化或追加数据
            if tracking_number not in result_list:
                result_list[tracking_number] = {"item": []}

            result_list[tracking_number]["item"].append(order_item)

        return result_list

    @classmethod
    def send_data(cls, code, result_list):
        # 获取当前时间
        desc = datetime.now().strftime("%Y-%m")

        result = {}

        for tracking_number, order_data in result_list.items():
            resp = cls.create_asn(code, desc, tracking_number, order_data["item"])
            print(f"跟踪号：{tracking_number}, 入库单号：{resp.get('receiving_code')}")
            result[tracking_number] = resp.get('receiving_code')

        return result

    @classmethod
    def save_excel(cls, file, result):
        # 读取 Excel 文件
        data = pd.read_excel(file)

        # 通过映射方式直接添加新列 'Receiving code'
        data['Receiving code'] = data['Shipment Request ID'].map(result)

        # 保存修改后的数据到原文件或另一个文件
        output_file = file.replace('.xlsx', '_updated.xlsx')
        data.to_excel(output_file, index=False)
        print(f"结果已保存到 {output_file}")

# 仓库编码和名称的映射
warehouse_map = {
    "USLAX01": "美西CA仓库",
    "KHDCN": "国内仓-练习使用",
    "USNJ01": "美东NJ仓库",
    "USWA03": "Sumner仓",
    "NOVILAND": "Noviland",
    "USHOU01": "美南TX仓库",
    "JPTOKYO01": "日本关东仓库",
    "CNFUJ01": "CNFUJ01",
}

if __name__ == '__main__':
    for warehouse_code, warehouse_name in warehouse_map.items():
        print(f"仓库编码：{warehouse_code} , 名称：{warehouse_name}")

    while True:
        # 提示用户输入
        warehouse_code = input("\n请输入仓库编码：").strip()
        # 根据输入查找仓库名称
        warehouse_name = warehouse_map.get(warehouse_code)
        if warehouse_name:
            break
        else:
            print(f"仓库编码 [{warehouse_code}] 未找到对应的仓库名称，请重新输入。")

    print(f"\n仓库编码 [{warehouse_code}] 对应的仓库名称是：{warehouse_name}")

    # 提示用户输入文件路径
    file_path = input("请输入 Excel 文件路径：").strip()

    process = Process()
    # 处理文件
    data_list = process.read_data(file_path)
    # 发送数据
    send_result = process.send_data(warehouse_code, data_list)
    # 保存结果到 Excel 文件
    process.save_excel(file_path, send_result)
