# coding: utf-8
import os
import threading
import queue
import time
import traceback
from tkinter import filedialog, simpledialog, messagebox
from DrissionPage import ChromiumPage
from dotenv import load_dotenv
import ttkbootstrap as ttk
from ttkbootstrap.constants import *
from app.helper import file, domain
from app.logger.gui import GuiLog
from app.vc.payment import Payment
from app.vc.payment_push import PaymentPush
from app.vc.return_goods import ReturnGoods
from app.vc.spa import Spa


class VCManagerGUI(ttk.Window):
    def __init__(self, logger):
        super().__init__(themename="cosmo")

        self.run_btn = None
        self.file_entry = None
        self.action_var = None
        self.shop_entry = None
        self.country_var = None
        self.log_text = None
        self.title("店铺管理工具")
        self.geometry("700x800")

        # 设置窗口属性禁止缩放
        self.resizable(False, False)

        # 初始化状态变量
        self.page = None
        self.running = False
        self.log_queue = queue.Queue()
        self.payee_code = None

        # 设置日志处理
        logger.set_console(self.log_queue)
        self.logger = logger

        # 设置窗口尺寸并居中
        self._center_window()

        # 创建界面组件
        self.create_widgets()

        # 启动日志处理循环
        self.after(100, self.process_log_queue)

    def _center_window(self):
        """设置窗口居中"""
        window_width = 700
        window_height = 800

        # 获取屏幕尺寸
        screen_width = self.winfo_screenwidth()
        screen_height = self.winfo_screenheight()

        # 计算居中坐标
        x = (screen_width - window_width) // 2
        y = (screen_height - window_height) // 2

        # 设置窗口位置
        self.geometry(f"{window_width}x{window_height}+{x}+{y}")

    def create_widgets(self):
        """创建主界面布局"""
        main_frame = ttk.Frame(self)
        main_frame.pack(fill=BOTH, expand=True, padx=15, pady=15)

        # 配置区域
        config_frame = ttk.Labelframe(main_frame, text="配置参数", padding=10)
        config_frame.pack(fill=X, pady=10)

        # 国家选择
        country_frame = ttk.Frame(config_frame)
        country_frame.grid(row=0, column=0, sticky=W, pady=5)
        ttk.Label(country_frame, text="国家:", width=6).pack(side=LEFT)
        self.country_var = ttk.StringVar(value="US")
        countries = [
            ("美国", "US"),
            ("英国", "UK"),
            ("日本", "JP"),
            ("法国", "FR"),
            ("德国", "DE"),
            ("加拿大", "CA"),
        ]
        for i, (text, value) in enumerate(countries):
            rb = ttk.Radiobutton(
                country_frame,
                text=text,
                variable=self.country_var,
                value=value
            )
            rb.pack(side=LEFT, padx=(10 if i != 0 else 0))

        # 功能选择
        action_frame = ttk.Frame(config_frame)
        action_frame.grid(row=1, column=0, sticky=W, pady=5)
        ttk.Label(action_frame, text="功能:", width=6).pack(side=LEFT)
        self.action_var = ttk.StringVar()
        actions = [
            ("SPA查询", "spa"),
            ("退货查询", "return"),
            # ("回款查询", "payment"),
            ("ERP回款", "payment_erp")
        ]
        for i, (text, value) in enumerate(actions):
            rb = ttk.Radiobutton(
                action_frame,
                text=text,
                variable=self.action_var,
                value=value
            )
            rb.pack(side=LEFT, padx=(10 if i != 0 else 0))

        # 文件选择区域
        file_frame = ttk.Labelframe(main_frame, text="数据文件", padding=10)
        file_frame.pack(fill=X, pady=10)

        self.file_entry = ttk.Entry(file_frame)
        self.file_entry.pack(side=LEFT, fill=X, expand=True, padx=5)
        ttk.Button(file_frame, text="浏览", command=self.select_file, width=8).pack(side=LEFT)

        # 控制按钮
        btn_frame = ttk.Frame(main_frame)
        btn_frame.pack(fill=X, pady=15)
        self.run_btn = ttk.Button(btn_frame, text="开始执行", command=self.start_process,
                                  style=PRIMARY, width=12)
        self.run_btn.pack(side=LEFT, padx=5)
        ttk.Button(btn_frame, text="清除日志", command=self.clear_log, width=10).pack(side=LEFT, padx=5)
        ttk.Button(btn_frame, text="退出", command=self.quit_app, width=8).pack(side=RIGHT)

        # 日志显示
        log_frame = ttk.Labelframe(main_frame, text="操作日志", padding=10)
        log_frame.pack(fill=BOTH, expand=True)

        self.log_text = ttk.ScrolledText(log_frame, state=DISABLED)
        self.log_text.pack(fill=BOTH, expand=True)

    def select_file(self):
        """选择数据文件"""
        path = filedialog.askopenfilename(filetypes=[("Excel文件", "*.xlsx *.xls")])
        if path:
            self.file_entry.delete(0, ttk.END)
            self.file_entry.insert(0, path)

    def start_process(self):
        """启动处理线程"""
        if self.running:
            messagebox.showwarning("警告", "已有任务正在运行")
            return

        if not self.validate_input():
            return

        self.running = True
        self.run_btn.config(state=ttk.DISABLED)
        threading.Thread(target=self.run_process, daemon=True).start()

    def validate_input(self):
        """验证输入有效性"""
        if not self.file_entry.get():
            messagebox.showerror("输入错误", "请选择数据文件")
            return False
        if not self.action_var.get():
            messagebox.showerror("输入错误", "请选择要执行的功能")
            return False
        return True

    def run_process(self):
        """后台处理流程"""
        try:
            # 初始化浏览器
            self.log("初始化浏览器引擎...")
            self.init_browser()

            # 获取参数
            params = {
                'country': self.country_var.get(),
                'action': self.action_var.get(),
                'file_name': self.file_entry.get()
            }

            # 特殊参数处理
            if params['action'] == "payment":
                self.get_payee_code()
                if not self.payee_code:
                    return

            # 创建处理器实例
            processor = self.create_processor(params)

            # 切换域名
            domain.domain_page(self.logger, self.page, self.country_var.get())

            # 执行核心操作
            self.log(f"开始执行 {params['action']} 操作...")
            processor.run(params['file_name'])
            processor.push_data_queue()
            self.log("操作成功完成！")

        except Exception as e:
            self.log(f"发生错误：{str(e)}")
            self.log(traceback.format_exc())
        finally:
            self.cleanup_resources()

    def init_browser(self):
        """初始化浏览器配置"""
        self.page = ChromiumPage()
        self.page.set.load_mode.normal()
        self.page.set.when_download_file_exists('overwrite')
        download_path = os.path.join(os.getcwd())
        self.page.set.download_path(download_path)
        self.log(f"下载目录设置为：{download_path}")

    def create_processor(self, params):
        """创建功能处理器"""

        # 根据国家编码获取店铺代码
        country = params['country']
        shop_code = "VECELO"
        if country == "JP":
            shop_code = "JP-VC"
        elif country == "UK":
            shop_code = "UK-VC"
        elif country == "DE":
            shop_code = "DE-VC"
        elif country == "FR":
            shop_code = "FR-VC"
        elif country == "CA":
            shop_code = "CA-VC"

        action = params['action']
        if action == "payment":
            return Payment(self.logger, self.page, country, self.payee_code, shop_code)
        elif action == "payment_erp":
            return PaymentPush(self.logger, country, shop_code)
        elif action == "return":
            return ReturnGoods(self.logger, self.page, country, shop_code)
        elif action == "spa":
            return Spa(self.logger, self.page, country, shop_code)
        else:
            raise ValueError(f"未知的功能类型：{action}")

    def get_payee_code(self):
        """获取回款Code"""
        self.after(0, self._show_payee_dialog)
        while self.running and not self.payee_code:
            time.sleep(0.1)

    def _show_payee_dialog(self):
        """显示回款Code输入对话框"""
        self.payee_code = simpledialog.askstring(
            "回款信息",
            "请输入回款Code（从URL参数获取）:",
            parent=self,
            initialvalue="VECET"
        )
        if not self.payee_code:
            self.log("用户取消回款Code输入")
            self.running = False

    def log(self, message):
        """记录日志到队列"""
        self.logger.info(message)
        # timestamp = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
        # self.log_queue.put(f"[{timestamp}] {message}")

    def process_log_queue(self):
        """处理日志队列"""
        while not self.log_queue.empty():
            msg = self.log_queue.get()
            self.log_text.configure(state=ttk.NORMAL)
            self.log_text.insert(ttk.END, msg + "\n")
            self.log_text.configure(state=ttk.DISABLED)
            self.log_text.see(ttk.END)
        self.after(100, self.process_log_queue)

    def clear_log(self):
        """清除日志"""
        self.log_text.configure(state=ttk.NORMAL)
        self.log_text.delete(1.0, ttk.END)
        self.log_text.configure(state=ttk.DISABLED)

    def cleanup_resources(self):
        """清理资源"""
        if self.page:
            try:
                self.page.close()
                self.log("浏览器资源已释放")
            except Exception as e:
                self.log(f"释放浏览器资源时出错：{str(e)}")
            finally:
                self.page = None
        self.running = False
        self.after(0, lambda: self.run_btn.config(state=ttk.NORMAL))

    def quit_app(self):
        """安全退出程序"""
        if messagebox.askokcancel("退出", "确认要退出程序吗？"):
            self.cleanup_resources()
            self.destroy()


if __name__ == "__main__":
    try:
        log = GuiLog()
        load_dotenv()
        app = VCManagerGUI(log)
        app.mainloop()
    except KeyboardInterrupt:
        exit(1)
