import pika
import json


class RabbitMQClient:
    def __init__(self, host='localhost',port='5672', username='guest', password='guest'):
        """
        初始化 RabbitMQ 客户端，创建连接和通道。
        :param host: RabbitMQ 主机地址
        :param username: RabbitMQ 账号
        :param password: RabbitMQ 密码
        """

        self.host = host
        self.port = port
        self.username = username
        self.password = password

        self.connection = None
        self.channel = None

        self.exchange = None
        self.queue = None
        self.routing_key = None

        self._initialize_connection()

    def _initialize_connection(self):
        """初始化连接和通道"""
        try:
            credentials = pika.PlainCredentials(self.username, self.password)
            self.connection = pika.BlockingConnection(pika.ConnectionParameters(
                host=self.host,port=self.port, credentials=credentials
            ))
            self.channel = self.connection.channel()
            print(f"Initialized connection to RabbitMQ at {self.host}.")
        except Exception as e:
            print(f"Error initializing RabbitMQ connection: {e}")
            raise

    def connect(self, queue, exchange='', exchange_type='direct', routing_key=''):
        """
        绑定队列和交换机
        :param queue: 队列名称
        :param exchange: 交换机名称
        :param exchange_type: 交换机类型
        :param routing_key: 路由键
        """
        try:
            # 声明交换机（如果有）
            if exchange:
                self.channel.exchange_declare(exchange=exchange, exchange_type=exchange_type, durable=True)

            # 声明队列
            self.channel.queue_declare(queue=queue, durable=True)

            # 绑定队列到交换机（如果有）
            if exchange:
                self.channel.queue_bind(queue=queue, exchange=exchange, routing_key=routing_key)

            self.queue = queue
            self.exchange = exchange
            self.routing_key = routing_key

            print(f"Connected to queue '{queue}' with exchange '{exchange}' (type: {exchange_type}).")
        except Exception as e:
            print(f"Error connecting to queue and exchange: {e}")
            raise

    def send_message(self, message, routing_key=None):
        """
        发送消息到绑定的交换机
        :param message: 要发送的消息
        :param routing_key: 路由键（如果不传，则使用绑定时的 routing_key）
        """
        routing_key = routing_key or self.routing_key

        try:
            if isinstance(message, dict):
                message = json.dumps(message)

            self.channel.basic_publish(
                exchange=self.exchange,
                routing_key=routing_key,
                body=message,
                properties=pika.BasicProperties(delivery_mode=2)
            )
            # print(f"Sent message: {message}")
        except Exception as e:
            print(f"Error sending message: {e}")

    def receive_message(self, callback):
        """
        接收消息并处理
        :param callback: 处理消息的回调函数
        """
        try:
            self.channel.basic_consume(queue=self.queue, on_message_callback=callback, auto_ack=True)
            print("Waiting for messages...")
            self.channel.start_consuming()
        except Exception as e:
            print(f"Error receiving message: {e}")

    def close(self):
        """关闭连接"""
        if self.connection and not self.connection.is_closed:
            self.connection.close()
            print("Connection closed.")