# coding: utf-8
import json
import queue
import threading
from datetime import datetime
from tkinter import filedialog, scrolledtext

import pandas as pd
import requests
import ttkbootstrap as ttk
import xmltodict
from ttkbootstrap.constants import *


# 原有的YcClient和Process类重构为实例方法
class YcClient:
    def __init__(self, app_key, app_token):
        self.app_key = app_key
        self.app_token = app_token
        self.base_url = "http://8.210.223.221/default/svc/web-service"

    def call_service(self, service, params_json):
        """调用SOAP服务的封装函数"""
        payload = f"""<?xml version="1.0" encoding="UTF-8"?>
            <SOAP-ENV:Envelope xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/" xmlns:ns1="http://www.example.org/Ec/">
                <SOAP-ENV:Body>
                    <ns1:callService>
                        <appToken>{self.app_token}</appToken>
                        <appKey>{self.app_key}</appKey>
                        <service>{service}</service>
                        <paramsJson>{json.dumps(params_json)}</paramsJson>
                    </ns1:callService>
                </SOAP-ENV:Body>
            </SOAP-ENV:Envelope>"""

        response = requests.post(
            self.base_url,
            data=payload,
            headers={"Content-Type": "text/xml;charset=UTF-8"}
        )

        if response.status_code != 200:
            raise Exception(f"HTTP错误: {response.status_code}, 内容: {response.text}")

        response_dict = xmltodict.parse(response.text)
        resp = (response_dict.get("SOAP-ENV:Envelope", {})
                .get("SOAP-ENV:Body", {})
                .get("ns1:callServiceResponse", {})
                .get('response', {}))

        json_res = json.loads(resp) if isinstance(resp, str) else resp

        if json_res.get("ask", "Failure") != "Success":
            raise Exception(json_res.get("message", "未知错误"))

        return json_res.get("data", {})

    def get_products(self, sku: str) -> dict:
        """获取商品列表"""
        params_json = {"pageSize": 10, "page": 1, "product_sku": sku}
        return self.call_service("getProductList", params_json)

    def create_asn(self, warehouse_code, desc, tracking_number, order_item_list):
        """创建入库单"""
        params_json = {
            "reference_no": tracking_number,
            "warehouse_code": warehouse_code,
            "tracking_number": tracking_number,
            "bulk_cargo_type": 1,
            "pallet_cnt": 1,
            "receiving_desc": desc,
            "items": order_item_list
        }
        return self.call_service("createAsn", params_json)


class Process(YcClient):
    def __init__(self, app_key, app_token):
        super().__init__(app_key, app_token)
        self.log_message = None

    @staticmethod
    def read_data(file):
        """读取并处理Excel数据"""
        result = pd.read_excel(file)
        result = result.groupby(['Shipment Request ID', 'SKU'], as_index=False).agg({'Return quantity': 'sum'})

        result_list = {}
        for _, item in result.iterrows():
            tracking_number = item['Shipment Request ID']
            order_item = {
                "product_sku": item['SKU'],
                "quantity": item['Return quantity'],
                "box_no": 1
            }
            result_list.setdefault(tracking_number, {"item": []})["item"].append(order_item)
        return result_list

    def send_data(self, warehouse_code, data_list):
        """发送数据到API"""
        desc = datetime.now().strftime("%Y-%m")
        result = {}
        for tracking_number, order_data in data_list.items():
            try:
                resp = self.create_asn(warehouse_code, desc, tracking_number, order_data["item"])
                result[tracking_number] = resp.get('receiving_code')
                self.log_message(f"跟踪号：{tracking_number}, 入库单号：{resp.get('receiving_code')}")
            except Exception as e:
                self.log_message(f"错误: {str(e)}")
        return result

    @staticmethod
    def save_excel(file, result):
        """保存结果到Excel"""
        data = pd.read_excel(file)
        data['Receiving code'] = data['Shipment Request ID'].map(result)
        output_file = file.replace('.xlsx', '_updated.xlsx')
        data.to_excel(output_file, index=False)
        return output_file


class Application(ttk.Window):
    def __init__(self):
        super().__init__(themename="cosmo")
        self.log_area = None
        self.run_btn = None
        self.file_entry = None
        self.warehouse_combo = None
        self.title("入库单创建工具")
        self.geometry("600x500")
        self.resizable(False, False)

        # 仓库编码映射
        self.warehouse_map = {
            "USLAX01": "美西CA仓库",
            "KHDCN": "国内仓-练习使用",
            "USNJ01": "美东NJ仓库",
            "USWA03": "Sumner仓",
            "NOVILAND": "Noviland",
            "USHOU01": "美南TX仓库",
            "JPTOKYO01": "日本关东仓库",
            "CNFUJ01": "CNFUJ01",
        }

        # 设置窗口尺寸并居中
        self._center_window()

        # 创建日志队列
        self.log_queue = queue.Queue()
        self.create_widgets()
        self.after(100, self.process_log_queue)

    def _center_window(self):
        """设置窗口居中"""
        window_width = 600
        window_height = 500

        # 获取屏幕尺寸
        screen_width = self.winfo_screenwidth()
        screen_height = self.winfo_screenheight()

        # 计算居中坐标
        x = (screen_width - window_width) // 2
        y = (screen_height - window_height) // 2

        # 设置窗口位置
        self.geometry(f"{window_width}x{window_height}+{x}+{y}")

    def create_widgets(self):
        """创建界面组件"""
        main_frame = ttk.Frame(self)
        main_frame.pack(fill=BOTH, expand=True, padx=10, pady=10)

        # 仓库选择
        warehouse_frame = ttk.Labelframe(main_frame, text="仓库选择", padding=10)
        warehouse_frame.pack(fill=X, pady=5)

        self.warehouse_combo = ttk.Combobox(
            warehouse_frame,
            values=[f"{code} - {name}" for code, name in self.warehouse_map.items()],
            state="readonly"
        )
        self.warehouse_combo.pack(fill=X, padx=5)

        # 文件选择
        file_frame = ttk.Labelframe(main_frame, text="Excel文件", padding=10)
        file_frame.pack(fill=X, pady=5)

        self.file_entry = ttk.Entry(file_frame)
        self.file_entry.pack(side=LEFT, fill=X, expand=True, padx=5)
        ttk.Button(file_frame, text="浏览", command=self.select_file).pack(side=LEFT, padx=5)

        # 控制按钮
        btn_frame = ttk.Frame(main_frame)
        btn_frame.pack(fill=X, pady=10)
        self.run_btn = ttk.Button(btn_frame, text="开始执行", command=self.start_process, style=PRIMARY)
        self.run_btn.pack(pady=5)

        # 日志显示
        log_frame = ttk.Labelframe(main_frame, text="操作日志", padding=10)
        log_frame.pack(fill=BOTH, expand=True)

        self.log_area = scrolledtext.ScrolledText(log_frame, state=DISABLED)
        self.log_area.pack(fill=BOTH, expand=True)

    def select_file(self):
        """选择Excel文件"""
        path = filedialog.askopenfilename(filetypes=[("Excel文件", "*.xlsx *.xls")])
        if path:
            self.file_entry.delete(0, END)
            self.file_entry.insert(0, path)

    def start_process(self):
        """启动处理线程"""
        app_key = "c906cc46bda8cea593c0b6b20eadb1f2"
        app_token = "2832175c4ee5c6efd79e129e919f2dfd"
        warehouse = self.warehouse_combo.get().split(" - ")[0]
        file_path = self.file_entry.get().strip()

        if not warehouse:
            self.log("错误：请选择仓库")
            return
        if not file_path:
            self.log("错误：请选择Excel文件")
            return

        self.run_btn.config(state=DISABLED)
        threading.Thread(
            target=self.process_data,
            args=(app_key, app_token, warehouse, file_path),
            daemon=True
        ).start()

    def process_data(self, app_key, app_token, warehouse, file_path):
        """后台数据处理线程"""
        try:
            processor = Process(app_key, app_token)
            processor.log_message = self.log  # 传递日志方法

            self.log("开始处理数据...")
            data_list = processor.read_data(file_path)

            if not data_list:
                self.log("没有需要处理的数据")
                return

            self.log("开始创建入库单...")
            result = processor.send_data(warehouse, data_list)

            self.log("正在保存结果文件...")
            output_path = processor.save_excel(file_path, result)

            self.log(f"处理完成！结果已保存至：{output_path}")

        except Exception as e:
            self.log(f"处理出错：{str(e)}")
        finally:
            self.run_btn.config(state=NORMAL)

    def log(self, message):
        """线程安全的日志记录"""
        self.log_queue.put(message)

    def process_log_queue(self):
        """处理日志队列"""
        while not self.log_queue.empty():
            msg = self.log_queue.get()
            self.log_area.config(state=NORMAL)
            self.log_area.insert(END, msg + "\n")
            self.log_area.config(state=DISABLED)
            self.log_area.see(END)
        self.after(100, self.process_log_queue)


if __name__ == "__main__":
    app = Application()
    app.mainloop()
