# coding: utf-8
# 回款明细

import pandas as pd
from app.vc import log, rabbit
from app.vc.interface import AutoInterface


class PaymentPush(AutoInterface):

    def __init__(self, country: str, shop_code: str):
        self.country = country
        self.shop_code = shop_code

    @staticmethod
    def __read_data(file_name):
        df = pd.read_excel(file_name, header=None)

        # 定位标题行
        pay_title = df[df[0].str.contains('Remittance payments', case=False, na=False)].index[0]
        inv_title = df[df[0].str.contains('Invoices', case=False, na=False)].index[0]

        # 定位表头起始行（跳过标题后的空行）
        pay_header = df.loc[pay_title + 1:].notna().any(axis=1).idxmax()
        inv_header = df.loc[inv_title + 1:].notna().any(axis=1).idxmax()

        # 计算第一个表格的结束位置（第二个标题前的空行）
        empty_lines = df.index[df.isnull().all(axis=1)].tolist()
        separator = max([x for x in empty_lines if pay_header < x < inv_title], default=inv_title - 1)

        # 读取并清理数据
        test = separator - pay_header - 1
        payments = pd.read_excel(file_name, header=pay_header, nrows=test).dropna(how='all')
        invoices = pd.read_excel(file_name, header=inv_header).dropna(how='all')

        return [payments, invoices]

    def push_data_queue(self):
        pass

    def run(self, file_name: str):
        payments, invoices = self.__read_data(file_name)

        # 将 'Payment Number' 列设置为索引
        payments.set_index('Payment Number', inplace=True)

        # 转换为字典，orient='index' 表示以索引为键
        payments_map = payments.to_dict(orient='index')

        log.info(f"共计：{len(invoices)} 订单")

        rabbit.connection()
        rabbit.connect(queue='refund_robot', routing_key='refund_robot', exchange='reports')

        i = 0
        for _, data in invoices.iterrows():
            i += 1
            payment_number = data.get("Payment Number")
            payment_date = payments_map.get(payment_number, {}).get('Payment Date', '')
            platform_payable_amount = data.get('Invoice Amount', '')
            if self.country == 'FR' or self.country == 'UK':
                platform_payable_amount = data.get('Net Amount Paid', '')

            push_data = {
                'payment_number': data.get('Payment Number', ''),  # 订单id
                'order_date': str(data.get('Invoice Date', '')),  # 发票时间
                'payment_date': str(payment_date),
                'order_no': data.get('Invoice Number', 0),  # 订单号
                'payment_type': data.get('Description', ''),  # Description
                'platform_payable_amount': platform_payable_amount,  # 平台应付金额
                'fee_amount': data.get("Terms Discount Taken", ''),  # 手续费
                'actual_payment': data.get('Amount Paid', ''),  # 实际支付金额
                'currency': data.get('Invoice Currency', ''),  # 货币
                'shop_code': self.shop_code,  # 店铺code
            }

            # 推送数据
            rabbit.send_message(push_data)

        rabbit.close()
