import os.path

import pandas as pd
import argparse


class InvoiceIDComparator:
    def __init__(self, file_a, file_b, invoice_column_name):
        self.file_a = file_a
        self.file_b = file_b
        self.invoice_column_name = invoice_column_name

    def get_invoice_ids_from_excel(self, file_path):
        """从Excel文件中获取所有sheet的Invoice ID"""
        excel_file = pd.ExcelFile(file_path)
        invoice_ids = set()  # 使用集合去重

        for sheet_name in excel_file.sheet_names:
            # 读取每个sheet的内容
            df = excel_file.parse(sheet_name)

            # 确保指定的列存在
            if self.invoice_column_name in df.columns:
                invoice_ids.update(df[self.invoice_column_name].dropna().unique())

            invoice_ids.add(sheet_name)  # 将sheet名也加入到集合中

        return invoice_ids

    def compare_invoice_ids(self):
        """比较两个Excel文件中的Invoice ID"""
        # 获取文件A中的Invoice ID和所有sheet名称
        invoice_ids_a = self.get_invoice_ids_from_excel(self.file_a)

        # 获取文件B中的Invoice ID和所有sheet名称
        invoice_ids_b = self.get_invoice_ids_from_excel(self.file_b)

        only_in_a = invoice_ids_a - invoice_ids_b
        only_in_b = invoice_ids_b - invoice_ids_a

        # 输出比较结果
        print("文件A中存在，但文件B中没有的 Invoice IDs:")
        print(only_in_a)

        print("\n文件B中存在，但文件A中没有的 Invoice IDs:")
        print(only_in_b)


def main():
    # 设置命令行参数
    parser = argparse.ArgumentParser(description="比较两个Excel文件中的Invoice ID差异")
    parser.add_argument('--original_file', default="ContraCogsInvoices.xls", help="原文件路径")
    parser.add_argument('--result_file', default="result.xls", help="结果文件路径")
    parser.add_argument('--invoice_column', default='Invoice ID', help="Invoice ID列的名称")

    # 解析命令行参数
    args = parser.parse_args()

    if os.path.exists(args.original_file) is False:
        raise FileExistsError("源文件不存在")

    if os.path.exists(args.result_file) is False:
        raise FileExistsError("结果文件不存在")


    # 创建 InvoiceIDComparator 实例并进行比较
    comparator = InvoiceIDComparator(args.original_file, args.result_file, args.invoice_column)
    comparator.compare_invoice_ids()


# 程序入口
if __name__ == "__main__":
    try:
        main()
    except Exception as e:
        print(e)