# coding: utf-8
# 退货
import os
from datetime import datetime

import pandas as pd
from DrissionPage import ChromiumPage as Page
from DrissionPage.errors import ElementNotFoundError

from app.helper import domain, api, excel, file
from app.logger.logger import Logger
from app.vc import rabbit
from app.vc.interface import AutoInterface


class ReturnGoods(AutoInterface):
    def __init__(self,logger: Logger, page: Page, country: str,shop_code: str):
        self.logger = logger
        self.page = page
        self.country = country
        self.shop_code = shop_code

        # 获取当前日期和时间并格式化
        current_datetime = datetime.now().strftime('%Y-%m-%d-%H-%M')
        # 原文件名
        file_name = "退货明细.xlsx"
        # 拼接新的文件名
        self.result_file_name = f"{current_datetime}_{self.country}_{file_name}"

    def __page_get(self, url):
        host = domain.switch_domain(self.country)
        full_url = host + url
        self.page.get(full_url, timeout=5)

    def __export_item_read_data(self, return_id: str):
        file_name = f"return_goods\\{return_id}.xls"
        if not os.path.isfile(file_name):
            while True:
                try:
                    # 打开退回详情下载明细
                    self.__page_get(f"katalmonsapp/vendor/members/returns/{return_id}")
                    self.page.ele("#file-download-button").click.to_download(rename=file_name)
                    file.wait_for_downloads(file_name)
                    break
                except ElementNotFoundError:
                    self.logger.warning("元素未找到,刷新网页")
                    self.page.refresh()

        # 读取回退商品详情
        return pd.read_excel(file_name)

    def run(self, file_name: str):
        if not os.path.isfile(file_name):
            raise FileNotFoundError(f"{file_name},文件不存在")

        # 读取sku映射关系
        relations_dict = api.sku_relations(self.country)

        # 读取list数据
        list_data = pd.read_excel(file_name)
        self.logger.info(f"共计：{len(list_data)} 订单")

        new_list_data = []
        i = 0
        for _, data in list_data.iterrows():
            i += 1
            return_id = data.get('Return ID')
            self.logger.info({"index": i, "return_id": return_id})

            # 下载退货详情表格读取数据
            item_data = self.__export_item_read_data(return_id)
            # 按 'Purchase order' 和 'ASIN' 分组，并对 'Quantity' 和 Total amount 进行求和
            item_data_result = item_data.groupby(['Purchase order', 'ASIN', 'Reason'], as_index=False).agg({
                'Quantity': 'sum',
                'Total amount': 'sum',
            })

            for _, item_row in item_data_result.iterrows():
                relation = relations_dict.get(item_row.get('ASIN'))
                erp_sku = relation.get('erp_sku', "")
                data_dict = data.to_dict()
                data_dict.update({
                    'Return Date': data_dict['Return Date'].strftime('%m/%d/%Y'),
                    'Return ID': str(data_dict['Return ID']),
                    'PO': item_row.get('Purchase order', ""),
                    'ASIN': item_row.get('ASIN', ""),
                    'SKU': erp_sku,
                    'Quantity': item_row.get('Quantity', 0),
                    # 替换回会数量和金额为详情里面的值
                    'Return quantity': item_row.get('Quantity', 0),  # 替换回会数量
                    'Reason': item_row.get('Reason', ""),
                    'Total cost': item_row.get('Total amount', 0),  # 替换金额
                    'Group Name': relation.get("name", ""),
                    'Group Code': relation.get("code", ""),
                })
                # 追加数据
                new_list_data.append(data_dict)

        excel.save_xls(new_list_data, self.result_file_name)

    def push_data_queue(self):
        rabbit.connection()
        rabbit.connect(queue='return_robot', routing_key='return_robot', exchange='reports')

        data = pd.read_excel(self.result_file_name, keep_default_na=False, na_values=[])

        for _, item_row in data.iterrows():
            push_data = {
                'return_id': item_row.get('Return ID', ''),
                'asin': item_row.get('ASIN', ''),  # ASIN
                'order_no': item_row.get('Purchase order', ''),  # 订单号
                'sku_quantity': item_row.get('Quantity', 0),  # 退回数量
                'sku_amount': item_row.get('Total cost', 0),  # Total cost
                'currency': item_row.get('Currency code', ''),  # Currency code
                'data_date': str(item_row.get('Return Date', '')),  # Return Date
                'erp_sku': item_row.get("SKU", ''),  # ERP SKU  # SKU1匹配
                'shop_code': self.shop_code,  # 店铺code
                'supplier_code': item_row.get('Vendor code', ''),  # 供应商编码
                'group_name': item_row.get('Group Name', ""),  # 组别 运营一组 运营二组
                'group_code': item_row.get('Group Code', ""),  # 组别 T1 T2
            }

            # 推送数据
            rabbit.send_message(push_data)

        rabbit.close()