# coding: utf-8

import pandas as pd

from lxml import etree
from app.vc import rabbit
from app.helper import domain, file
from app.logger.logger import Logger
from app.vc.interface import AutoInterface
from DrissionPage import ChromiumPage as Page
from datetime import datetime

class AdvertCost(AutoInterface):
    def __init__(self, logger: Logger, page: Page, country: str):
        self.logger = logger
        self.page = page
        self.country = country

        # 获取当前日期和时间并格式化
        current_datetime = datetime.now().strftime('%Y-%m-%d-%H-%M')
        # 原文件名
        file_name = "广告数据.xlsx"
        # 拼接新的文件名
        self.result_file_name = f"{current_datetime}_{file_name}"

    def __page_get(self, url):
        host = domain.switch_domain(self.country)
        full_url = host + url
        self.page.get(full_url, timeout=5)

    @staticmethod
    def __parse_vendor_info(xml_content):
        # 解析 XML 内容
        parser = etree.XMLParser(recover=True)
        tree = etree.fromstring(xml_content, parser=parser)

        # 存储结果
        results = []

        # 查找所有 h5 元素
        h5_elements = tree.xpath('//h5[@class="heading align-start color-normal heading-size--normal"]')

        # 查找所有包含 Account ID 的 p 元素
        p_elements = tree.xpath(
            '//div[@class="container vendor-account-id     a20m border-type-none background-color-transparent desktop"]/p[@class="text align-start color-normal text--word-break--normal text-size--normal"]')

        # 查找所有 button 元素的 onclick URL
        button_elements = tree.xpath('//button[@class="form-button a20m-form-button-type-submit"]')

        # 确保 h5、p 和 button 元素数量匹配
        for h5, p, button in zip(h5_elements, p_elements, button_elements):
            # 提取 onclick 中的 URL
            onclick = button.get('onclick', '')
            url = onclick.replace("window.location.href='", "").rstrip("'") if onclick else ""

            results.append({
                'name': h5.text.strip(),
                'id': p.text.strip().replace('Account ID: ', ''),
                'url': url
            })

        return results

    def run(self,file_name: str):
        self.__page_get("hz/vendor/members/advertising/home?ref_=vc_xx_subNav")
        wrapper_html = self.page.ele(". columns-wrapper a20m-columns-wrapper", timeout=5).html
        # 解析并提取信息
        vendor_info = self.__parse_vendor_info(wrapper_html)

        all_data = []

        # 打印结果
        for info in vendor_info:
            name = info['name']
            self.logger.debug(f"开始处理小组:{name}")

            entity_id = info['id'][1:]

            host = domain.switch_ad_domain(self.country)
            self.page.get(f"{host}cm/products?entityId=ENTITY{entity_id}")

            # 判断是否正常
            target_text = "There’s an issue with your payment method and your campaigns have stopped delivering."
            text_found = self.page.ele(f'text:{target_text}')
            if text_found:
                self.logger.warning(f"{name} 异常")
                continue

            # 点击日历
            self.page.ele("#products_view:products_view:dateRangeFilter:openContainer").click()
            self.page.wait(1)

            # 点击昨天选项
            yesterday_but = self.page.ele('xpath://button[@value="YESTERDAY"]')
            if yesterday_but:
                yesterday_but.click()
                self.page.wait(3)
                # 点击导出
                date_format = datetime.now().strftime('%Y%m%d%H%M')
                file_name = f"products\\{date_format}-{info['id']}.csv"
                self.page.ele("#products_view:products_view:export").click.to_download(rename=file_name)
                file.wait_for_downloads(file_name)
                df = pd.read_csv(file_name)
                if len(df) == 0:
                    continue
                all_data.append(df)

        # 合并所有 DataFrame
        combined_df = pd.concat(all_data, ignore_index=True)
        # 写入最终文件
        combined_df.to_excel(self.result_file_name, index=False)

    def push_data_queue(self):
        rabbit.connection()
        rabbit.connect(queue='advertising', routing_key='advertising', exchange='reports')

        data = pd.read_excel(self.result_file_name, keep_default_na=False, na_values=[])

        for _, item_row in data.iterrows():
            asin = item_row.get('Products', '').split("-")[0]
            spend = item_row.get('Spend(USD)', ''),  # 金额

            if self.country == 'JP':
                asin = item_row.get('商品', '').split("-")[0]
                spend = item_row.get('花费(JPY)', ''),  # 金额

            push_data = {
                'asin': asin,  # ASIN
                'spend': spend,  # 金额
                'timestamp': int(datetime.now().timestamp()),
                'country': self.country,
            }
            # 推送数据
            rabbit.send_message(push_data)
        rabbit.close()