# coding: utf-8
# spa
import math
import os
from datetime import datetime

import pandas as pd
from DrissionPage import ChromiumPage as Page
from DrissionPage.errors import ElementNotFoundError
from lxml import etree

from app.helper import domain, file, excel, api, helper
from app.logger.logger import Logger
from app.vc import rabbit
from app.vc.interface import AutoInterface


class Spa(AutoInterface):
    def __init__(self, logger: Logger, page: Page, country: str, shop_code: str):
        self.logger = logger
        self.page = page
        self.country = country
        self.shop_code = shop_code

        # 获取当前日期和时间并格式化
        current_datetime = datetime.now().strftime('%Y-%m-%d-%H-%M')
        file_name = "spa.xlsx"
        self.result_file_name = f"{current_datetime}_{self.country}_{file_name}"

    def __page_get(self, url):
        host = domain.switch_domain(self.country)
        full_url = host + url
        self.page.get(full_url, timeout=10)

    def __get_report_table_links(self, invoice_id):
        retry_count = 0
        while True:
            if retry_count > 5:
                self.logger.warning(f"{invoice_id} 获取报表链接重试次数超过{retry_count}次,请检查发票号")
                return None

            try:
                self.__page_get(f"hz/vendor/members/coop?searchText={invoice_id}")
                # 点击选项卡
                self.page.ele("#a-autoid-2-announce").click()
                self.page.wait(2)
                # 点击下载报表选项
                # self.page.ele(f"#invoiceDownloads-{invoice_id}_2").click()
                self.page.ele('xpath://li[a[contains(text(), "Backup report")]]/a').click()
                self.page.wait(2)

                # 获取报表表单内容
                report_table_html = self.page.ele("#backup-report-table").html
                if report_table_html is None or report_table_html == "":
                    self.logger.warning("表单内容为空,刷新网页")
                    self.page.refresh()
                    continue

                tree = etree.HTML(report_table_html)
                links = tree.xpath('//table[@id="backup-report-table"]//a/@href')
                if len(links) == 0:
                    self.logger.warning(f"{invoice_id} 未获取到下载连接，开始重试...")
                    retry_count += 1
                    continue

                return links
            except ElementNotFoundError:
                self.logger.warning(f"{invoice_id} 元素未找到,刷新网页")
                self.page.refresh()

    def __export_item_read_data(self, invoice_id):
        file_name = f"spa\\{invoice_id}.xlsx"
        if os.path.isfile(file_name):
            df = pd.read_excel(file_name, sheet_name=None)
            return df

        # 获取报表表单所有链接
        links = self.__get_report_table_links(invoice_id)
        if len(links) == 0:
            # data_list = get_report_agreement_text(invoice_id)
            # return {"Accrual For Current Period": pd.DataFrame(data_list)}
            self.logger.warning(f"{invoice_id} 没有下载连接")
            return None

        for link in links:
            host = domain.switch_domain(self.country).rstrip('/')
            report_file = os.path.join(os.getcwd(), "BackupReport.xls")
            while True:
                file_link = host + link
                self.page.get(file_link, retry=0)
                is_down = file.wait_for_downloads(report_file, 20)
                if is_down: break
                self.logger.warning(f"下载 {invoice_id} 失败,重新下载")

            try:
                df = pd.read_excel(report_file)
                # 获取表头
                headers = df.columns.tolist()
                # 要检查的列名
                column_names_to_check = ["Rebate In Agreement Currency", "Vendor Funding In Agreement Currency"]
                # 判断头文件是否满足读取条件,不满足删除文件夹
                header_is_normal = any(column in headers for column in column_names_to_check)
                if not header_is_normal:
                    os.remove(report_file)
                    self.logger.warning(f"{invoice_id} 文件格式不满足要求")
                    continue

                save_report_file = os.path.join(os.getcwd(), "spa", str(invoice_id), "BackupReport.xls")
                file.move_file(report_file, save_report_file)
                # 创建 ExcelFile 对象
                excel_file = pd.ExcelFile(save_report_file)
                # 获取所有工作表名称
                sheet_names = excel_file.sheet_names
                for sheet_name in sheet_names:
                    df = pd.read_excel(save_report_file, sheet_name=sheet_name)
                    data = df[df['Asin'].notna()]
                    excel.save_xls(data, file_name, sheet_name)
                return pd.read_excel(file_name, sheet_name=None)
            except ValueError:
                self.logger.warning(f"{invoice_id} 文件格式错误")

        return None

    def __process_large_items(self, item_list, relation_data, coop, va_tax):
        """处理大数据列表 (item_list 长度 >= 10)"""

        processed_items = []
        for _, item in item_list.iterrows():
            asin = item.get('Asin', None)
            if not self.__validate_asin(asin):
                continue

            relation = relation_data.get(asin, {})
            rebate = item.get("Rebate In Agreement Currency", None)
            vendor_funding = item.get("Vendor Funding In Agreement Currency", None)
            original_balance = rebate or vendor_funding

            funding_type = coop.get('Funding Type', "")
            if funding_type == "":
                funding_type = coop.get('Funding type', "")

            processed_item = item.copy()
            processed_item.pop("Asin")
            processed_item["Agreement title"] = coop.get("Agreement title")
            processed_item["Invoice date"] = coop.get("Invoice date")
            processed_item['Funding Type'] = funding_type
            processed_item['Asin'] = asin
            processed_item['ERP SKU'] = relation.get("erp_sku")
            processed_item['Group Name'] = relation.get("name")
            processed_item['Group Code'] = relation.get("code")
            processed_item["Original balance"] = original_balance
            processed_item["vat_tax"] = va_tax
            processed_item['vat_rate'] = item.get("VAT rate", 0)
            processed_items.append(processed_item)

        return processed_items

    def __process_small_items(self, item_list, relation_data, coop, vat_tax):
        """处理小数据列表 (item_list 长度 < 10)"""
        processed_items = []
        for _, item in item_list.iterrows():
            asin = item.get('Asin', None)
            if asin is None:
                asin = item.get('ASIN', None)
            if not self.__validate_asin(asin):
                continue

            relation = relation_data.get(asin, {})
            rebate = item.get("Rebate In Agreement Currency", None)
            vendor_funding = item.get("Vendor Funding In Agreement Currency", None)
            original_balance = rebate or vendor_funding

            processed_item = coop.copy()
            processed_item.pop("Original balance")
            processed_item.pop("Invoice date")
            if "Funding Type" in processed_item.keys():
                processed_item.pop("Funding Type")
                funding_type = coop.get('Funding Type', "")
            else:
                processed_item.pop("Funding type")
                funding_type = coop.get('Funding type', "")

            processed_item["Invoice date"] = coop.get("Invoice date")
            processed_item['Funding Type'] = funding_type
            processed_item["Order Date"] = item.get("Order Date")
            processed_item['Purchase Order'] = relation.get("Purchase Order")
            processed_item["Agreement Currency"] = item.get("Agreement Currency")
            processed_item["Asin"] = asin
            processed_item["ERP SKU"] = relation.get("erp_sku")
            processed_item["Group Name"] = relation.get("name")
            processed_item['Group Code'] = relation.get("code")
            processed_item["Original balance"] = original_balance
            processed_item["vat_tax"] = vat_tax
            processed_item['vat_rate'] = item.get("VAT rate", 0)
            processed_items.append(processed_item)
        return processed_items

    @staticmethod
    def __validate_asin(asin):
        """验证 ASIN 是否有效"""
        return asin and not (isinstance(asin, float) and math.isnan(asin))

    def __write_sheet(self, writer, data, sheet_name):
        if not isinstance(sheet_name, str):
            sheet_name = str(sheet_name)

        self.logger.info(f"开始写入 {sheet_name}, 共计 {len(data)} 条")
        df = pd.DataFrame(data)  # 将数据转换为 DataFrame
        df.to_excel(writer, sheet_name=sheet_name, index=False)

    def __save_excel(self, sheet_data, large_sheet_data, new_file_name):
        """保存数据到 Excel 文件"""
        # 初始化 Excel 写入器
        with pd.ExcelWriter(new_file_name) as writer:
            # 写入小数据
            if sheet_data:
                self.logger.info(f"保存小数据，共计 {len(sheet_data)} 条")
                self.__write_sheet(writer, sheet_data, "Sheet1")

            # 写入大数据（使用多线程并行写入不同表）
            if large_sheet_data:
                self.logger.info(f"保存大数据，共计 {sum(len(data) for data in large_sheet_data.values())} 条")
                for sheet_name, data in large_sheet_data.items():
                    self.__write_sheet(writer, data, sheet_name)
                # with ThreadPoolExecutor() as executor:
                #     for sheet_name, data in large_sheet_data.items():
                #         executor.submit(write_sheet, writer, data, sheet_name)

    @staticmethod
    def __calculate_sheets_total_amount(sheets):
        """
        计算每个sheet总金额
        """
        total_amount = 0
        # 定义可能的列名（按优先级顺序）
        possible_columns = [
            "Rebate In Agreement Currency",
            "Vendor Funding In Agreement Currency"
        ]

        for sheet_name, df in sheets.items():
            # 确定要使用的列名
            rebate_column = None
            for col in possible_columns:
                if col in df.columns:
                    rebate_column = col
                    break

            # 找到第一个空行（NaN值）的索引
            first_empty = df[rebate_column].isna().idxmax()
            if pd.isna(first_empty) or first_empty == 0:
                total = df[rebate_column].sum()
                total_amount = total_amount + total
            else:
                # 只计算第一个空行之前的数据
                valid_data = df.loc[:first_empty - 1, rebate_column]
                # 计算总和，忽略NaN值
                total_amount = total_amount + valid_data.sum()

        return round(total_amount, 2)

    def push_data_queue(self):
        self.logger.info("开始读取数据....")
        # 读取Excel文件
        xls = pd.ExcelFile(self.result_file_name)
        self.logger.info("开始推送消息....")

        rabbit.connection()
        rabbit.connect(queue='spa_robot', routing_key='spa_robot', exchange='reports')

        total_rows = 0
        total_amount = 0.00

        for sheet_name in xls.sheet_names:
            self.logger.info(f"开始处理 {sheet_name}")
            df = pd.read_excel(xls, sheet_name, keep_default_na=False, na_values=[])

            if sheet_name == "Sheet1":
                # 根据Invoice ID去重，保留第一条记录
                total_rows = len(df['Invoice ID'].drop_duplicates())
                # 清洗Original balance列，去除$和千位分隔符
                df['Original balance'] = (df['Original balance'].
                                                 astype(str).
                                                 str.
                                                 replace(r'[\$,]', '',regex=True))
                # 转换为数值型，处理非数值数据
                df['Original balance'] = pd.to_numeric(df['Original balance'], errors='coerce')
                # 计算总和，忽略NaN
                total_amount += df['Original balance'].sum(skipna=True)
            else:
                total_rows += 1

                target_column = None
                if 'Rebate In Agreement Currency' in df.columns:
                    target_column = 'Rebate In Agreement Currency'
                elif 'Vendor Funding In Agreement Currency' in df.columns:
                    target_column = 'Vendor Funding In Agreement Currency'

                if target_column:
                    # 转换为数值型，处理非数值数据
                    df[target_column] = pd.to_numeric(df[target_column], errors='coerce')
                    rebate_total = df[target_column].sum(skipna=True)
                    total_amount += rebate_total

            parent_id = sheet_name

            for _, item_row in df.iterrows():
                if sheet_name == "Sheet1":
                    parent_id = item_row.get('Invoice ID', "")

                # 协议类型
                agreement_type = 0
                agreement_title = item_row.get("Agreement title")
                if "4.22%" in agreement_title:
                    agreement_type = 3
                if "5.5%" in agreement_title:
                    agreement_type = 2
                if "10.0%" in agreement_title:
                    agreement_type = 1

                if 'Funding Type' in item_row.keys():
                    funding_type = item_row.get('Funding Type', "")
                else:
                    funding_type = item_row.get('Funding type', "")

                push_data = {
                    'ad_date': item_row.get('Invoice date', ""),  # spa费用数据日期
                    'erp_sku': item_row.get('ERP SKU', ""),  # ERP SKU
                    'ad_amount': item_row.get('Original balance', ""),  # spa费用金额
                    'ad_amount_currency': item_row.get('Agreement Currency', ""),  # spa费用币制
                    'funding_type': funding_type,  # 资金类型
                    'group_name': item_row.get('Group Name', ""),  # 组别 运营一组 运营二组
                    'group_code': item_row.get('Group Code', ""),  # 组别 T1 T2
                    'asin': item_row.get('Asin', ""),  # ASIN
                    'shop_code': self.shop_code,  # 店铺code
                    'type': 2,  # 1 sheet1 2 其他sheet
                    'parent_id': parent_id,  # sheet1 为Invoice ID 其他sheet为sheet名称
                    'order_no': item_row.get('Purchase Order', ""),  # 订单号
                    'agreement_title': agreement_type,
                    'vat_tax': item_row.get('vat_tax', 0),
                    'vat_rate': item_row.get("vat_rate", 0)
                }
                # 推送数据
                rabbit.send_message(push_data)

        rabbit.close()

        self.logger.info(f"所有sheet的总行数: {total_rows}")
        self.logger.info(f"所有sheet的总金额: {total_amount}")

    def run(self, file_name: str):
        # 获取数据
        relation_data = api.sku_relations(self.country)  # 获取 ASIN 与 SKU 的对应关系数据

        if not os.path.isfile(file_name):
            raise FileNotFoundError(f"{file_name},文件不存在")

        # 读取数据列表
        coop_list = pd.read_excel(file_name)
        self.logger.info(f"共计: {len(coop_list)} 条数据")

        sheet_data = []  # 用于保存小数据
        large_sheet_data = {}  # 保存大数据（需要分 Sheet）
        # 遍历合作列表
        for index, coop in coop_list.iterrows():
            index += 1
            invoice_id = coop.get("Invoice ID")  # 获取发票 ID
            self.logger.info({"index": index, "invoice_id": invoice_id})

            if not invoice_id:
                self.logger.warning(f"缺少 Invoice ID，跳过第 {index} 条数据")
                continue

            # 获取当前发票的 item 列表
            item_dict = self.__export_item_read_data(invoice_id)
            if item_dict is None:
                sheet_data.append(coop)
                self.logger.warning(f"{invoice_id} 暂无报告信息")
                continue

            # 计算每个sheet总金额
            total_amount = self.__calculate_sheets_total_amount(item_dict)
            original_balance = coop.get("Original balance", 0.0)
            original_balance = helper.extract_numeric_value(original_balance, self.country)
            # 0 没有税 1 有税
            vat_tax = int(total_amount != original_balance)

            if len(item_dict) > 1:
                for i, value in enumerate(item_dict.values(), start=1):
                    invoice_id_num = f"{invoice_id}-{i}"
                    processed_items = self.__process_large_items(value, relation_data, coop, vat_tax)
                    if processed_items:
                        large_sheet_data[invoice_id_num] = processed_items
            else:
                item_list = item_dict.get('Accrual For Current Period')
                # 如果是列表且长度 >= 10 则新增sheet
                if len(item_list) >= 10:
                    processed_items = self.__process_large_items(item_list, relation_data, coop, vat_tax)
                    if processed_items:
                        large_sheet_data[invoice_id] = processed_items
                else:
                    # 如果是较小的列表
                    processed_items = self.__process_small_items(item_list, relation_data, coop, vat_tax)
                    sheet_data.extend(processed_items)

        # 保存数据到 Excel 文件
        self.__save_excel(sheet_data, large_sheet_data, self.result_file_name)

        self.logger.info(f"文件 {self.result_file_name} 保存完成，路径：{os.path.abspath(self.result_file_name)}")
