# coding: utf-8
import re
import time
from datetime import datetime

import pandas as pd
from DrissionPage import ChromiumPage as Page
from DrissionPage.errors import ElementNotFoundError
from lxml import etree

from app.helper import domain, file
from app.logger.logger import Logger
from app.vc import rabbit
from app.vc.interface import AutoInterface


class AdvertCost(AutoInterface):
    def __init__(self, logger: Logger, page: Page, country: str):
        super().__init__()  # 调用父类初始化
        self.logger = logger
        self.page = page
        self.country = country
        # 运行状态
        self.running = False
        self.select_day = 0

        # 获取当前日期和时间并格式化
        current_datetime = datetime.now().strftime('%Y-%m-%d-%H-%M')
        # 原文件名
        file_name = "广告数据.xlsx"
        # 拼接新的文件名
        self.result_file_name = f"{current_datetime}_{file_name}"

    def set_running(self, running: bool):
        self.running = running

    def __page_get(self, url):
        host = domain.switch_domain(self.country)
        full_url = host + url
        self.page.get(full_url, timeout=5)

    @staticmethod
    def __parse_vendor_info(xml_content):
        # 解析 XML 内容
        parser = etree.XMLParser(recover=True)
        tree = etree.fromstring(xml_content, parser=parser)

        # 存储结果
        results = []

        # 查找所有 h5 元素
        h5_elements = tree.xpath('//h5[@class="heading align-start color-normal heading-size--normal"]')

        # 查找所有包含 Account ID 的 p 元素
        p_elements = tree.xpath(
            '//div[@class="container vendor-account-id     a20m border-type-none background-color-transparent desktop"]/p[@class="text align-start color-normal text--word-break--normal text-size--normal"]')

        # 查找所有 button 元素的 onclick URL
        button_elements = tree.xpath('//button[@class="form-button a20m-form-button-type-submit"]')

        # 确保 h5、p 和 button 元素数量匹配
        for h5, p, button in zip(h5_elements, p_elements, button_elements):
            # 提取 onclick 中的 URL
            onclick = button.get('onclick', '')
            url = onclick.replace("window.location.href='", "").rstrip("'") if onclick else ""

            results.append({
                'name': h5.text.strip(),
                'id': p.text.strip().replace('Account ID: ', ''),
                'url': url
            })

        return results

    def __click_button(self):
        # 点击过滤按钮
        option_selector = '#products_view:products_view:addFilterSelector:optionSelector'
        self.page.wait.eles_loaded(option_selector)
        self.page.ele(option_selector).click()
        time.sleep(2)

        # 点击spend选项
        spend = '#products_view:products_view:addFilterSelector:option-spend'
        self.page.wait.eles_loaded(spend)
        self.page.ele(spend).click()
        time.sleep(2)

        # 点击应用
        apply = '#products_view:products_view:filter:spend:apply'
        self.page.wait.eles_loaded(apply)  # 等待元素加载
        self.page.ele(apply).click()
        time.sleep(5)

        # 判断分页按钮是否存在
        try:
            pagination_element = self.page.ele('#products_view:products_view:pagination-dropdown', timeout=5).html
            if pagination_element is None or pagination_element == "":
                self.logger.debug("分页按钮不存在，跳过翻页")
                return
        except ElementNotFoundError as e:
            self.logger.debug(f"分页按钮不存在或查找失败，跳过翻页: {str(e)}")
            return

        # 点击分页按钮
        dropdown = '#products_view:products_view:pagination-dropdown'
        self.page.wait.eles_loaded(dropdown)
        self.page.ele(dropdown).click()
        time.sleep(1)

        # 选择200条
        dropdown200 = '#products_view:products_view:pagination-dropdown-200'
        self.page.wait.eles_loaded(dropdown200)
        self.page.ele(dropdown200).click()
        time.sleep(5)

        # 翻页
        pagination_selector = '#products_view:products_view:pagination-next'
        self.page.wait.eles_loaded(pagination_selector)
        pagination_btn = self.page.ele(pagination_selector)

        iteration = 0
        while True:
            # 检查 disabled 属性
            if pagination_btn.attr('disabled') is None:
                pagination_btn.click()
                self.logger.debug(f"第 {iteration + 1} 页已点击，等待加载...")
                # 等待页面加载完成
                time.sleep(3)
                self.page.wait.eles_loaded(pagination_selector, timeout=10)
                iteration += 1
                continue
            else:
                self.logger.debug("分页按钮已禁用，停止翻页")
                break

    @staticmethod
    def __parse_date(text:str):
        # 匹配两种格式的通用模式
        match = re.search(r'Date range:\s*(\w+)\s+(\w+),?\s*(\d{4})', text)
        if not match:
            raise ValueError("无法解析日期")

        part1, part2, year = match.groups()

        # 判断哪个是月份（包含字母的是月份）
        if part1.isdigit():
            # 格式: "3 Dec 2025"
            day, month_str = part1, part2
            date_str = f"{day} {month_str} {year}"
            fmt = "%d %b %Y"
        else:
            # 格式: "Jan 15, 2024"
            month_str, day = part1, part2
            date_str = f"{month_str} {day}, {year}"
            fmt = "%b %d, %Y"

        return int(datetime.strptime(date_str, fmt).timestamp())

    def run(self, file_name: str):
        self.__page_get("hz/vendor/members/advertising/home?ref_=vc_xx_subNav")
        # wrapper_html = self.page.ele(". columns-wrapper a20m-columns-wrapper", timeout=5).html
        # 解析并提取信息
        # vendor_info = self.__parse_vendor_info(wrapper_html)

        all_data = []
        country_store_ids = {
            "US": [
                "A1N0INZWHQEC1X", "A1JFHJEJBHFH4W", "A1P4BWBIUQH4X6",
                "A2VLKUSRNEY0D6", "A1OC20P16QN7BY", "A1YZS56TASXGM1", "A1EC618IVW7B1L"
            ],
            "JP": ["A2S5HH4XRRCAZ0", "A2JCA95F83WKJL"],
            "UK": ["AK7T8OM0L2CTO"],
            "FR": ["A1LAUA4LDUDQH9"],
            "DE": ["A1JJUIIUQRO63M"],
            "CA": ["A3OE9R2606Z2UP"]
        }

        store_ids = country_store_ids[self.country]

        for store_id in store_ids:
            # 检查暂停状态
            self.check_pause()

            name = store_id
            self.logger.debug(f"开始处理店铺:{name}")

            entity_id = store_id[1:]

            # 替换域名前缀
            host = domain.switch_domain(self.country).replace("vendorcentral", "advertising")
            self.page.get(f"{host}cm/products?entityId=ENTITY{entity_id}")

            # 判断是否正常
            target_text = "There’s an issue with your payment method and your campaigns have stopped delivering."
            text_found = self.page.ele(f'text:{target_text}')
            if text_found:
                self.logger.warning(f"{name} 异常")
                continue

            # 点击日历
            self.page.ele("#products_view:products_view:dateRangeFilter:openContainer").click()
            self.page.wait(1)

            # 点击昨天选项
            yesterday_but = self.page.ele('xpath://button[@value="YESTERDAY"]')
            if yesterday_but:
                yesterday_but.click()
                self.page.wait(3)

                # 找到按钮元素
                btn = self.page.ele('#products_view:products_view:dateRangeFilter:openContainer')
                # 获取按钮的文本
                timestamp = self.__parse_date(btn.text)
                # 解析成 datetime转换成时间戳
                self.select_day = timestamp

                # 点击分页按钮
                self.__click_button()

                # 点击导出
                date_format = datetime.now().strftime('%Y%m%d%H%M')
                file_name = f"products\\{date_format}-{name}.csv"
                self.page.ele("#products_view:products_view:export").click.to_download(rename=file_name)
                file.wait_for_downloads(file_name)
                df = pd.read_csv(file_name)
                if len(df) == 0:
                    continue
                all_data.append(df)

        # 合并所有 DataFrame
        combined_df = pd.concat(all_data, ignore_index=True)
        # 写入最终文件
        combined_df.to_excel(self.result_file_name, index=False)

    def push_data_queue(self):
        rabbit.connection()
        rabbit.connect(queue='advertising', routing_key='advertising', exchange='reports')

        data = pd.read_excel(self.result_file_name, keep_default_na=False, na_values=[])

        # 定义国家到货币符号的映射
        currency_symbols = {
            "US": "USD",
            "UK": "GBP",
            "JP": "JPY",
            "FR": "EUR",
            "DE": "EUR",
            "CA": "CAD",
        }

        for _, item_row in data.iterrows():
            symbol = currency_symbols.get(self.country)
            asin = item_row.get('Products', '').split("-")[0]
            spend = item_row.get(f'Spend({symbol})', 0.00),  # 金额
            if asin == '' or spend == 0.00:
                self.logger.warning(f"asin为空或无金额")
                continue

            if isinstance(spend, tuple):
                spend = float(spend[0])

            push_data = {
                'asin': asin,  # ASIN
                'spend': spend,  # 金额
                'timestamp': self.select_day,
                'country': self.country,
            }
            # 推送数据
            rabbit.send_message(push_data)
        rabbit.close()