# coding: utf-8
# 回款明细
import os
import urllib.parse
import warnings

import pandas as pd
from DrissionPage import ChromiumPage
from DrissionPage.errors import PageDisconnectedError, ElementNotFoundError

from helper import helper, excel

page = ChromiumPage()
page.set.load_mode.normal()
page.set.when_download_file_exists('overwrite')
# 下载目录
download_path = os.getcwd()
# 检查下载目录是否存在，如果不存在则创建
helper.make_dir(download_path)
# 设置下载路径，确保在打开浏览器前设置
page.set.download_path(download_path)

# 忽略 openpyxl 样式警告
warnings.filterwarnings("ignore", category=UserWarning, module="openpyxl")


def export_list_read_data():
    file_name = 'Payments.xlsx'
    if not os.path.isfile(file_name):
        page.get(f"https://vendorcentral.amazon.com/hz/vendor/members/remittance/home")
        page.ele("#remittance-home-select-all").click()
        page.ele("#remittance-home-export-link").click.to_download().wait()

    df = pd.read_excel(file_name, skiprows=22)
    # 定义正则表达式模式，匹配包含 'Price' 或 'PCR' 或 'XXXXXXXX/XXXX/' 的描述
    pattern = r'Price|PCR|Missed|Shortage|^[A-Z0-9]{8}/[A-Z0-9]{4}/'
    # 过滤符合条件的行
    return df[df['Description'].str.contains(pattern, na=False, regex=True)]


def export_details_read_data(invoice_number):
    # 读取详情内容
    file_name = f"invoices\\{invoice_number}.csv"
    if not os.path.isfile(file_name):
        params = {
            "invoiceNumber": invoice_number,
            "payeeCode": "VECET",
            "activeTab": "lineItems",
        }
        # 将字典转换为 URL 查询参数
        query_string = urllib.parse.urlencode(params)
        page.get(
            f"https://vendorcentral.amazon.com/hz/vendor/members/inv-mgmt/invoice-details?" + query_string)
        try:
            page.ele("#line-items-export-to-spreadsheet-announce").click.to_download(rename=file_name).wait()
        except ElementNotFoundError:
            print("导出按钮不存在刷新网页")
            page.refresh()
            export_details_read_data(invoice_number)

    if not os.path.isfile(file_name):
        export_details_read_data(invoice_number)

    return pd.read_csv(file_name, skiprows=2, engine='python', on_bad_lines='skip')


def main():
    list_data = export_list_read_data()
    excel.save_xls(list_data, "回款数据.xlsx", "Remittance payments")

    all_normal_pay_data = []
    all_price_pay_data = []
    for _, data in list_data.iterrows():
        invoice_number = data.get("Invoice Number")
        description = data.get("Description")
        if "Price" in description or "PCR" in description or "Missed" in description or "Shortage" in description:
            # 获取前8位
            invoice_number = invoice_number[:8]

        print(invoice_number)

        # invoice_search_page = page.new_tab(f"https://vendorcentral.amazon.com/hz/vendor/members/inv-mgmt/invoice-po-search?searchByNumberToken={invoice_number}")
        # invoice_search_page.close()

        # 下载excel文件并读取数据
        detail_data = export_details_read_data(invoice_number)

        # 初始化列表存储新字段数据
        normal_pay_data = []
        price_pay_data = []

        for index, detail_datum in detail_data.iterrows():
            amount = detail_datum.get('Amount', 0)  # 使用默认值 0 防止 None
            code = "VECET"
            deduction_points = 1

            # 计算扣除后的金额
            amount_after_deduction = amount - (amount * deduction_points / 100)
            # 复制原始行数据，避免直接修改
            new_record = detail_datum.copy()
            new_record["IsFinished"] = "是"
            new_record["DeductionPoints"] = f"{deduction_points}%"  # 拼接百分号
            new_record["Code"] = code
            new_record["AmountAfterDeduction"] = amount_after_deduction

            # 将处理后的记录添加到临时列表
            normal_pay_data.append(new_record)
            # 将处理后的记录添加到临时列表
            price_pay_data.append(new_record)

        # 添加到汇总列表
        all_normal_pay_data.append(pd.DataFrame(normal_pay_data))
        all_price_pay_data.append(pd.DataFrame(price_pay_data))

    # 将所有数据合并为一个 DataFrame
    normal_pay_summary = pd.concat(all_normal_pay_data, ignore_index=True)
    price_pay_summary = pd.concat(all_price_pay_data, ignore_index=True)

    excel.save_xls(normal_pay_summary, "回款数据.xlsx", "正常回款导出明细")
    excel.save_xls(price_pay_summary, "回款数据.xlsx", "Price导出明细")


if __name__ == '__main__':
    try:
        main()
    except KeyboardInterrupt:
        pass
    except PageDisconnectedError as e:
        print("与页面的连接已断开")
