# coding: utf-8

import os
import threading
import queue
import time
import traceback
from tkinter import messagebox
from DrissionPage import ChromiumPage
import ttkbootstrap as ttk
from dotenv import load_dotenv
from ttkbootstrap.constants import *

from app.helper import domain, file
from app.logger.gui import GuiLog
from app.vc.advert_cost import AdvertCost
from app.vc.product_sales import ProductSales


class VCManagerGUI(ttk.Window):
    def __init__(self, logger):
        super().__init__(themename="cosmo")

        self.domain_login = None
        self.processor = None
        self.actions = None
        self.countries = None
        self.run_btn = None
        self.pause_btn = None
        self.shop_entry = None
        self.country_var = None
        self.action_var = None
        self.log_text = None
        self.title("广告费工具")
        self.geometry("520x600")

        # 设置窗口属性禁止缩放
        self.resizable(False, False)

        # 初始化状态变量
        self.page = None
        self.running = False
        self.paused = False
        self.log_queue = queue.Queue()
        self.payee_code = None

        # 设置日志处理
        logger.set_console(self.log_queue)
        self.logger = logger

        # 设置窗口尺寸并居中
        self._center_window()

        # 创建界面组件
        self.create_widgets()

        # 启动日志处理循环
        self.after(100, self.process_log_queue)  # type: ignore

    def _center_window(self):
        """设置窗口居中"""
        window_width = 540
        window_height = 600

        # 获取屏幕尺寸
        screen_width = self.winfo_screenwidth()
        screen_height = self.winfo_screenheight()

        # 计算居中坐标
        x = (screen_width - window_width) // 2
        y = (screen_height - window_height) // 2

        # 设置窗口位置
        self.geometry(f"{window_width}x{window_height}+{x}+{y}")

    def create_widgets(self):
        """创建主界面布局"""
        main_frame = ttk.Frame(self)
        main_frame.pack(fill='both', expand=True, padx=15, pady=15)

        # 配置区域
        config_frame = ttk.Labelframe(main_frame, text="配置参数", padding=10)
        config_frame.pack(fill='x', pady=10)

        # 国家选择
        country_frame = ttk.Frame(config_frame)
        country_frame.grid(row=0, column=0, sticky=W, pady=5)
        ttk.Label(country_frame, text="国家:", width=6).pack(side='left')
        self.country_var = ttk.StringVar(value="all")
        self.countries = [
            ("全部", "all"),
            ("美国", "US"),
            ("日本", "JP"),
            ("英国", "UK"),
            ("法国", "FR"),
            ("德国", "DE"),
            ("加拿大", "CA"),
        ]
        for i, (text, value) in enumerate(self.countries):
            rb = ttk.Radiobutton(country_frame, text=text, variable=self.country_var, value=value)
            rb.pack(side='left', padx=(10 if i != 0 else 0))

        # 功能选择
        action_frame = ttk.Frame(config_frame)
        action_frame.grid(row=1, column=0, sticky=W, pady=5)
        ttk.Label(action_frame, text="功能:", width=6).pack(side='left')
        self.action_var = ttk.StringVar(value="all")
        self.actions = [
            ("全部", "all"),
            ("广告费", "advert_cost"),
            ("商品销量", "product_sales")
        ]
        for i, (text, value) in enumerate(self.actions):
            rb = ttk.Radiobutton(action_frame, text=text, variable=self.action_var, value=value)
            rb.pack(side='left', padx=(10 if i != 0 else 0))

        # 控制按钮
        btn_frame = ttk.Frame(main_frame)
        btn_frame.pack(fill='x', pady=15)
        self.pause_btn = ttk.Button(btn_frame, text="暂停", command=self.toggle_pause,
                                    style='warning.TButton', width=12, state=ttk.DISABLED)
        self.pause_btn.pack(side='right', padx=5)
        self.run_btn = ttk.Button(btn_frame, text="开始执行", command=self.start_process,
                                  style='success.TButton', width=12)
        self.run_btn.pack(side='right', padx=5)

        # 日志显示
        log_frame = ttk.Labelframe(main_frame, text="操作日志", padding=10)
        log_frame.pack(fill='both', expand=True)

        self.log_text = ttk.ScrolledText(log_frame, state=DISABLED)
        self.log_text.pack(fill='both', expand=True)

    def start_process(self):
        """启动处理线程"""
        if self.running:
            messagebox.showwarning("警告", "已有任务正在运行")
            return

        if not self.action_var.get():
            messagebox.showerror("输入错误", "请选择要执行的功能")
            return

        self.running = True
        self.paused = False
        self.run_btn.config(text="执行中...", style='info.TButton', state=ttk.DISABLED)
        self.pause_btn.config(state=ttk.NORMAL)
        threading.Thread(target=self.run_process, daemon=True).start()

    def toggle_pause(self):
        """切换暂停/继续状态"""
        self.paused = not self.paused
        if self.paused:
            self.pause_btn.config(text="继续", style='success.TButton')
            self.logger.info("已暂停")
            if self.processor:
                self.processor.set_running(False)
            if self.domain_login:
                self.domain_login.set_status(False)
        else:
            self.pause_btn.config(text="暂停", style='warning.TButton')
            self.logger.info("已继续")
            if self.processor:
                self.processor.set_running(True)
            if self.domain_login:
                self.domain_login.set_status(True)

    def run_process(self):
        """后台处理流程"""
        try:
            self.logger.info("初始化浏览器引擎...")
            self.init_browser()
            country = self.country_var.get()
            action = self.action_var.get()

            # 处理国家选择
            countries = self.countries if country == "all" else [(None, country)]
            countries = [(text, value) for text, value in countries if value != "all"]

            # 先登录选择的国家
            for _, country_value in countries:
                self.domain_login = domain.LoginDomain(self.logger, self.page, country_value)
                self.domain_login.set_status(True)
                self.domain_login.login_check()

            for _, country_value in countries:
                # 处理动作选择
                actions = self.actions if action == "all" else [(None, action)]
                actions = [(text, value) for text, value in actions if value != "all"]
                for _, action_value in actions:
                    self.logger.info(f"开始执行 {action_value} 操作于国家 {country_value}...")
                    self.processor = self._get_processor(action_value, country_value)
                    if self.processor:
                        self.processor.set_running(True)
                        # self.processor.run("")
                        # self.processor.push_data_queue()

            self.logger.info("操作成功完成！")
        except Exception as e:
            self.logger.error(f"发生错误：{str(e)}")
            self.logger.error(traceback.format_exc())
        finally:
            self.cleanup_resources()

    def _get_processor(self, action, country):
        """根据动作类型返回处理器实例"""
        processors = {
            "advert_cost": AdvertCost,
            "product_sales": ProductSales
        }
        processor_class = processors.get(action)
        if processor_class:
            return processor_class(self.logger, self.page, str(country))
        return None

    def init_browser(self):
        """初始化浏览器配置"""
        self.page = ChromiumPage()
        self.page.set.load_mode.normal()
        self.page.set.when_download_file_exists('overwrite')
        download_path = os.path.join(os.getcwd())
        self.page.set.download_path(download_path)
        file.make_dir('products')

    def log(self, message):
        """记录日志到队列"""
        self.logger.info(message)
        timestamp = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
        self.log_queue.put(f"[{timestamp}] {message}")

    def process_log_queue(self):
        """处理日志队列"""
        while not self.log_queue.empty():
            msg = self.log_queue.get()
            self.log_text.configure(state=ttk.NORMAL)
            self.log_text.insert(ttk.END, msg + "\n")
            self.log_text.configure(state=ttk.DISABLED)
            self.log_text.see(ttk.END)
        self.after(200, self.process_log_queue)  # type: ignore

    def clear_log(self):
        """清除日志"""
        self.log_text.configure(state=ttk.NORMAL)
        self.log_text.delete(1.0, ttk.END)
        self.log_text.configure(state=ttk.DISABLED)

    def cleanup_resources(self):
        """清理资源"""
        if self.page:
            try:
                self.page.close()
                self.logger.info("浏览器资源已释放")
            except Exception as e:
                self.logger.info(f"释放浏览器资源时出错：{str(e)}")
            finally:
                self.page = None

        self.running = False
        self.paused = False
        self.run_btn.configure(text="开始执行", style='success.TButton')
        self.run_btn['state'] = ttk.NORMAL
        self.pause_btn.configure(text="暂停", style='warning.TButton')
        self.pause_btn['state'] = ttk.DISABLED

if __name__ == "__main__":
    try:
        load_dotenv()
        log = GuiLog()
        app = VCManagerGUI(log)
        app.mainloop()
    except KeyboardInterrupt:
        exit(1)