# coding: utf-8
# spa
import math
import os
import pandas as pd
from lxml import etree
from datetime import datetime
from urllib.parse import urlparse, parse_qs
from DrissionPage import ChromiumPage as Page
from DrissionPage.errors import ElementNotFoundError
from app.vc import log
from app.vc.interface import AutoInterface
from app.helper import domain, file, excel, rabbitmq, api


class Spa(AutoInterface):
    def __init__(self, page: Page, country: str, shop_code: str):
        self.page = page
        self.country = country
        self.shop_code = shop_code

        # 获取当前日期和时间并格式化
        current_datetime = datetime.now().strftime('%Y-%m-%d-%H-%M')
        file_name = "spa.xlsx"
        self.result_file_name = f"{current_datetime}_{self.country}_{file_name}"

    def __page_get(self, url):
        host = domain.switch_domain(self.country)
        full_url = host + url
        self.page.get(full_url, timeout=10)

    def __get_report_table_html(self, invoice_id):
        while True:
            try:
                self.__page_get(f"hz/vendor/members/coop?searchText={invoice_id}")
                # 点击选项卡
                self.page.ele("#a-autoid-2-announce").click()
                # 下载报表
                self.page.ele(f"#invoiceDownloads-{invoice_id}_2").click()
                self.page.wait(1)
                # 获取报表表单内容
                report_table_html = self.page.ele("#backup-report-table").html
                if report_table_html is None or report_table_html == "":
                    log.warning("表单内容为空,刷新网页")
                    self.page.refresh()
                    continue
                return report_table_html
            except ElementNotFoundError:
                log.warning("元素未找到,刷新网页")
                self.page.refresh()

    def __export_item_read_data(self, invoice_id):
        file_name = f"spa\\{invoice_id}.xlsx"
        if os.path.isfile(file_name):
            df = pd.read_excel(file_name, sheet_name=None)
            return df

        # 获取报表表单内容
        report_table_html = self.__get_report_table_html(invoice_id)
        tree = etree.HTML(report_table_html)
        # 提取所有链接
        links = tree.xpath('//table[@id="backup-report-table"]//a/@href')
        if len(links) == 0:
            # data_list = get_report_agreement_text(invoice_id)
            # return {"Accrual For Current Period": pd.DataFrame(data_list)}
            return None

        for link in links:
            # 解析链接中的查询参数
            parsed_url = urlparse(link)
            query_params = parse_qs(parsed_url.query)
            # 提取 filename 参数
            filename = query_params.get('fileName', ['未找到文件名'])[0]
            report_file_tmp_dir = os.getcwd() + f"\\spa\\{invoice_id}\\{filename}\\"
            host = domain.switch_domain(self.country)
            report_file = report_file_tmp_dir + "BackupReport.xls"
            while True:
                self.page.download(host + link, report_file_tmp_dir, show_msg=False)
                is_down = file.wait_for_downloads(report_file, 60)
                if is_down: break
                log.warning(f"下载 {invoice_id} 失败,重新下载")

            try:
                df = pd.read_excel(report_file)
                # 获取表头
                headers = df.columns.tolist()
                # 要检查的列名
                column_names_to_check = ["Rebate In Agreement Currency", "Vendor Funding In Agreement Currency"]
                # 判断头文件是否满足读取条件,不满足删除文件夹
                header_is_normal = any(column in headers for column in column_names_to_check)
                if not header_is_normal:
                    continue

                # 创建 ExcelFile 对象
                excel_file = pd.ExcelFile(report_file)
                # 获取所有工作表名称
                sheet_names = excel_file.sheet_names
                for sheet_name in sheet_names:
                    df = pd.read_excel(report_file, sheet_name=sheet_name)
                    data = df[df['Asin'].notna()]
                    excel.save_xls(data, file_name, sheet_name)
                return pd.read_excel(file_name, sheet_name=None)
            except ValueError:
                pass

    def __process_large_items(self, item_list, relation_data, coop):
        """处理大数据列表 (item_list 长度 >= 10)"""
        processed_items = []
        for _, item in item_list.iterrows():
            asin = item.get('Asin', None)
            if not self.__validate_asin(asin):
                continue

            relation = relation_data.get(asin, {})
            rebate = item.get("Rebate In Agreement Currency", None)
            vendor_funding = item.get("Vendor Funding In Agreement Currency", None)

            processed_item = item.copy()
            processed_item.pop("Title")
            processed_item.pop("Asin")
            processed_item["Invoice date"] = coop.get("Invoice date")
            processed_item['Funding Type'] = coop.get("Funding Type")
            processed_item['Asin'] = asin
            processed_item['ERP SKU'] = relation.get("erp_sku")
            processed_item['Group Name'] = relation.get("name")
            processed_item['Group Code'] = relation.get("code")
            processed_item["Original balance"] = rebate or vendor_funding
            processed_items.append(processed_item)

        return processed_items

    def __process_small_items(self, item_list, relation_data, coop):
        """处理小数据列表 (item_list 长度 < 10)"""
        processed_items = []
        for _, item in item_list.iterrows():
            asin = item.get('Asin', None)
            if asin is None:
                asin = item.get('ASIN', None)
            if not self.__validate_asin(asin):
                continue

            relation = relation_data.get(asin, {})
            rebate = item.get("Rebate In Agreement Currency", None)
            vendor_funding = item.get("Vendor Funding In Agreement Currency", None)

            processed_item = coop.copy()
            processed_item.pop("Agreement title")
            processed_item.pop("Original balance")
            processed_item.pop("Invoice date")
            processed_item.pop("Funding Type")

            processed_item["Invoice date"] = coop.get("Invoice date")
            processed_item['Funding Type'] = coop.get("Funding Type")
            processed_item["Order Date"] = item.get("Order Date")
            processed_item['Purchase Order'] = relation.get("Purchase Order")
            processed_item["Agreement Currency"] = item.get("Agreement Currency")
            processed_item["Asin"] = asin
            processed_item["ERP SKU"] = relation.get("erp_sku")
            processed_item["Group Name"] = relation.get("name")
            processed_item['Group Code'] = relation.get("code")
            processed_item["Original balance"] = rebate or vendor_funding
            processed_items.append(processed_item)
        return processed_items

    @staticmethod
    def __validate_asin(asin):
        """验证 ASIN 是否有效"""
        return asin and not (isinstance(asin, float) and math.isnan(asin))

    @staticmethod
    def __write_sheet(writer, data, sheet_name):
        if not isinstance(sheet_name, str):
            sheet_name = str(sheet_name)

        log.info(f"开始写入 {sheet_name}, 共计 {len(data)} 条")
        df = pd.DataFrame(data)  # 将数据转换为 DataFrame
        df.to_excel(writer, sheet_name=sheet_name, index=False)

    def __save_excel(self, sheet_data, large_sheet_data, new_file_name):
        """保存数据到 Excel 文件"""
        # 初始化 Excel 写入器
        with pd.ExcelWriter(new_file_name) as writer:
            # 写入小数据
            if sheet_data:
                log.info(f"保存小数据，共计 {len(sheet_data)} 条")
                self.__write_sheet(writer, sheet_data, "Sheet1")

            # 写入大数据（使用多线程并行写入不同表）
            if large_sheet_data:
                log.info(f"保存大数据，共计 {sum(len(data) for data in large_sheet_data.values())} 条")
                for sheet_name, data in large_sheet_data.items():
                    self.__write_sheet(writer, data, sheet_name)
                # with ThreadPoolExecutor() as executor:
                #     for sheet_name, data in large_sheet_data.items():
                #         executor.submit(write_sheet, writer, data, sheet_name)

        log.info(f"文件 {new_file_name} 保存完成，路径：{os.path.abspath(new_file_name)}")

    def push_data_queue(self):
        rabbit = rabbitmq.RabbitMQClient()
        rabbit.connect(queue='spa_robot', routing_key='spa_robot', exchange='reports')
        log.info("开始读取数据....")
        data_dict = pd.read_excel(self.result_file_name, sheet_name=None, keep_default_na=False, na_values=[])
        log.info("开始推送消息....")
        for sheet_name, values in data_dict.items():
            for _, item_row in values.iterrows():
                if sheet_name == "Sheet1":
                    parent_id = item_row.get('Invoice ID', "")
                else:
                    parent_id = sheet_name

                push_data = {
                    'ad_date': item_row.get('Invoice date', ""),  # spa费用数据日期
                    'erp_sku': item_row.get('ERP SKU', ""),  # ERP SKU
                    'ad_amount': item_row.get('Original balance', ""),  # spa费用金额
                    'ad_amount_currency': item_row.get('Agreement Currency', ""),  # spa费用币制
                    'funding_type': item_row.get('Funding Type', ""),  # 资金类型
                    'transaction_type': item_row.get('Transaction Type', ""),  # 交易类型
                    'group_name': item_row.get('Group Name', ""),  # 组别 运营一组 运营二组
                    'group_code': item_row.get('Group Code', ""),  # 组别 T1 T2
                    'asin': item_row.get('Asin', ""),  # ASIN
                    'shop_code': self.shop_code,  # 店铺code
                    'type': 2,  # 1 sheet1 2 其他sheet
                    'parent_id': parent_id,  # sheet1 为Invoice ID 其他sheet为sheet名称
                    'order_no': item_row.get('Purchase Order', ""),  # 订单号
                }
                # 推送数据
                rabbit.send_message(push_data)

    def run(self, file_name: str):
        # 获取数据
        relation_data = api.sku_relations(self.country)  # 获取 ASIN 与 SKU 的对应关系数据

        if not os.path.isfile(file_name):
            raise FileNotFoundError(f"{file_name},文件不存在")

        # 读取数据列表
        coop_list = pd.read_excel(file_name)
        log.info(f"共计: {len(coop_list)} 条数据")

        sheet_data = []  # 用于保存小数据
        large_sheet_data = {}  # 保存大数据（需要分 Sheet）
        # 遍历合作列表
        for index, coop in coop_list.iterrows():
            index += 1
            invoice_id = coop.get("Invoice ID")  # 获取发票 ID
            log.info({"index": index, "invoice_id": invoice_id})

            if not invoice_id:
                log.warning(f"缺少 Invoice ID，跳过第 {index} 条数据")
                continue

            # 获取当前发票的 item 列表
            item_dict = self.__export_item_read_data(invoice_id)
            if item_dict is None:
                sheet_data.append(coop)
                log.warning(f"{invoice_id} 暂无报告信息")
                continue

            if len(item_dict) > 1:
                for i, value in enumerate(item_dict.values(), start=1):
                    invoice_id_num = f"{invoice_id}-{i}"
                    processed_items = self.__process_large_items(value, relation_data, coop)
                    if processed_items:
                        large_sheet_data[invoice_id_num] = processed_items
            else:
                item_list = item_dict.get('Accrual For Current Period')
                # 如果是列表且长度 >= 10 则新增sheet
                if len(item_list) >= 10:
                    processed_items = self.__process_large_items(item_list, relation_data, coop)
                    if processed_items:
                        large_sheet_data[invoice_id] = processed_items
                else:
                    # 如果是较小的列表
                    processed_items = self.__process_small_items(item_list, relation_data, coop)
                    sheet_data.extend(processed_items)

        # 保存数据到 Excel 文件
        self.__save_excel(sheet_data, large_sheet_data, self.result_file_name)
