# coding: utf-8
"""
店铺管理工具 - 现代化 GUI 版本
支持 SPA 查询、退货查询、ERP 回款等功能
"""

import os
import threading
import queue
import time
import traceback
from datetime import datetime
from tkinter import filedialog, simpledialog, messagebox
from DrissionPage import ChromiumPage
from dotenv import load_dotenv
import pandas as pd
import ttkbootstrap as ttk
from ttkbootstrap.scrolled import ScrolledText
from ttkbootstrap.constants import *

from app.helper import domain
from app.logger.gui import GuiLog
from app.vc.payment import Payment
from app.vc.payment_push import PaymentPush
from app.vc.return_goods import ReturnGoods
from app.vc.spa import Spa


class VCManagerGUI:
    """店铺管理工具 GUI 类"""

    # 窗口配置 - 加高窗口以显示更多日志
    WINDOW_WIDTH = 620
    WINDOW_HEIGHT = 800

    # 国家配置
    COUNTRY_OPTIONS = [
        ("🇺🇸 美国", "US"),
        ("🇬🇧 英国", "UK"),
        ("🇯🇵 日本", "JP"),
        ("🇫🇷 法国", "FR"),
        ("🇩🇪 德国", "DE"),
        ("🇨🇦 加拿大", "CA"),
    ]

    # 功能配置
    ACTION_OPTIONS = [
        ("📊 SPA查询", "spa"),
        ("📦 退货查询", "return"),
        ("💰 ERP回款", "payment_erp"),
    ]

    # 推送配置
    PUSH_OPTIONS = [
        ("✅ 是", "1"),
        ("❌ 否", "0"),
    ]

    def __init__(self, logger):
        """初始化应用程序"""
        # 状态变量
        self.page = None
        self.running = False
        self.paused = False  # 暂停状态
        self.processor = None  # 当前处理器
        self.domain_login = None  # 域名登录对象
        self.log_queue = queue.Queue()
        self.payee_code = None
        self.total_count = 0  # 数据总条数
        self.current_progress = 0  # 当前进度

        # GUI 组件引用
        self.log_text = None
        self.country_var = None
        self.action_var = None
        self.push_msg_var = None
        self.file_entry = None
        self.progress_bar = None
        self.progress_label = None
        self.status_label = None
        self.run_btn = None
        self.pause_btn = None  # 暂停按钮
        self.stats_labels = {}

        # 设置日志
        logger.set_console(self.log_queue)
        self.logger = logger

        # 初始化主窗口 - 使用深色主题
        self.root = ttk.Window(themename="darkly")
        self.root.title("🛠️ 店铺管理工具")
        self.root.resizable(False, False)

        # 设置自定义样式
        self._setup_custom_styles()

        # 构建 GUI
        self._build_gui()

        # 窗口居中
        self._center_window()

        # 启动日志处理
        self.root.after(100, self._process_log_queue)

    @staticmethod
    def _setup_custom_styles():
        """设置自定义样式"""
        style = ttk.Style()

        # 标题样式
        style.configure(
            'Title.TLabel',
            font=('Segoe UI', 20, 'bold'),
            foreground='#ffffff'
        )

        # 副标题样式
        style.configure(
            'Subtitle.TLabel',
            font=('Segoe UI', 9),
            foreground='#a0a0a0'
        )

        # 区域标题样式
        style.configure(
            'SectionTitle.TLabel',
            font=('Segoe UI', 10, 'bold'),
            foreground='#74b9ff'
        )

        # 统计数字样式
        style.configure(
            'Stats.TLabel',
            font=('Segoe UI', 14, 'bold'),
            foreground='#00cec9'
        )

    def _center_window(self):
        """窗口居中"""
        screen_width = self.root.winfo_screenwidth()
        screen_height = self.root.winfo_screenheight()

        x = int((screen_width - self.WINDOW_WIDTH) / 2)
        y = int((screen_height - self.WINDOW_HEIGHT) / 3)

        self.root.geometry(f"{self.WINDOW_WIDTH}x{self.WINDOW_HEIGHT}+{x}+{y}")

    def _build_gui(self):
        """构建完整 GUI 界面"""
        # 主容器
        main_container = ttk.Frame(self.root, padding=12)
        main_container.pack(fill='both', expand=True)

        # 1. 标题区域 - 紧凑
        self._build_header(main_container)

        # 2. 配置卡片 - 紧凑
        self._build_config_card(main_container)

        # 3. 文件选择 - 紧凑
        self._build_file_card(main_container)

        # 4. 日志区域 - 主要区域，占用大部分空间
        self._build_log_area(main_container)

        # 5. 底部状态栏
        self._build_status_bar(main_container)

    @staticmethod
    def _build_header(parent):
        """构建标题区域 - 紧凑版"""
        header_frame = ttk.Frame(parent)
        header_frame.pack(fill='x', pady=(0, 8))

        # 主标题
        title_label = ttk.Label(
            header_frame,
            text="🛠️ 店铺管理工具",
            style='Title.TLabel'
        )
        title_label.pack()

    def _build_config_card(self, parent):
        """构建配置卡片区域 - 紧凑版"""
        # 卡片容器
        card_frame = ttk.LabelFrame(
            parent,
            text=" ⚙️ 配置参数 ",
            padding=8,
            bootstyle="info"
        )
        card_frame.pack(fill='x', pady=(0, 6))

        # === 国家选择区 ===
        self.country_var = ttk.StringVar(value="US")

        country_row = ttk.Frame(card_frame)
        country_row.pack(fill='x', pady=(0, 5))

        ttk.Label(
            country_row,
            text="📍 市场:",
            font=('Segoe UI', 9, 'bold'),
            foreground='#74b9ff'
        ).pack(side='left', padx=(0, 8))

        for i, (name, code) in enumerate(self.COUNTRY_OPTIONS):
            btn = ttk.Radiobutton(
                country_row,
                text=name,
                value=code,
                variable=self.country_var,
                bootstyle="info-toolbutton",
                width=8
            )
            btn.pack(side='left', padx=2)

        # === 功能选择区 ===
        self.action_var = ttk.StringVar(value="spa")

        action_row = ttk.Frame(card_frame)
        action_row.pack(fill='x', pady=(0, 5))

        ttk.Label(
            action_row,
            text="🎯 功能:",
            font=('Segoe UI', 9, 'bold'),
            foreground='#00b894'
        ).pack(side='left', padx=(0, 8))

        for i, (name, code) in enumerate(self.ACTION_OPTIONS):
            btn = ttk.Radiobutton(
                action_row,
                text=name,
                value=code,
                variable=self.action_var,
                bootstyle="success-toolbutton",
                width=10
            )
            btn.pack(side='left', padx=2)

        # === 推送选择区 ===
        self.push_msg_var = ttk.StringVar(value="1")

        push_row = ttk.Frame(card_frame)
        push_row.pack(fill='x')

        ttk.Label(
            push_row,
            text="📤 推送:",
            font=('Segoe UI', 9, 'bold'),
            foreground='#fdcb6e'
        ).pack(side='left', padx=(0, 8))

        for i, (name, code) in enumerate(self.PUSH_OPTIONS):
            btn = ttk.Radiobutton(
                push_row,
                text=name,
                value=code,
                variable=self.push_msg_var,
                bootstyle="warning-toolbutton",
                width=6
            )
            btn.pack(side='left', padx=2)

    def _build_file_card(self, parent):
        """构建文件选择卡片 - 紧凑版"""
        file_frame = ttk.LabelFrame(
            parent,
            text=" 📁 数据文件 ",
            padding=8,
            bootstyle="secondary"
        )
        file_frame.pack(fill='x', pady=(0, 6))

        # 文件选择行
        file_row = ttk.Frame(file_frame)
        file_row.pack(fill='x')

        self.file_entry = ttk.Entry(
            file_row,
            font=('Consolas', 9)
        )
        self.file_entry.pack(side='left', fill='x', expand=True, padx=(0, 8))

        browse_btn = ttk.Button(
            file_row,
            text="📂 浏览",
            command=self._select_file,
            bootstyle="outline",
            width=8
        )
        browse_btn.pack(side='right')

        # === 进度区域 ===
        progress_row = ttk.Frame(file_frame)
        progress_row.pack(fill='x', pady=(8, 0))

        self.progress_label = ttk.Label(
            progress_row,
            text="⏳ 准备就绪",
            font=('Segoe UI', 9),
            foreground='#b2bec3'
        )
        self.progress_label.pack(anchor='w')

        self.progress_bar = ttk.Progressbar(
            progress_row,
            mode='determinate',
            bootstyle="success-striped",
            maximum=100,
            value=0
        )
        self.progress_bar.pack(fill='x', pady=(3, 8))

        # === 操作按钮 ===
        button_frame = ttk.Frame(file_frame)
        button_frame.pack()

        self.run_btn = ttk.Button(
            button_frame,
            text="🚀 开始执行",
            command=self._start_process,
            bootstyle="success",
            width=14
        )
        self.run_btn.pack(side='left', padx=5)

        self.pause_btn = ttk.Button(
            button_frame,
            text="⏸ 暂停",
            command=self._toggle_pause,
            bootstyle="warning-outline",
            width=14,
            state='disabled'
        )
        self.pause_btn.pack(side='left', padx=5)

    def _build_log_area(self, parent):
        """构建日志显示区域 - 主要区域"""
        log_frame = ttk.LabelFrame(
            parent,
            text=" 📝 运行日志 ",
            padding=8,
            bootstyle="dark"
        )
        log_frame.pack(fill='both', expand=True, pady=(0, 5))

        # 使用 ScrolledText - 增大高度
        self.log_text = ScrolledText(
            log_frame,
            height=20,  # 增大日志区域高度
            autohide=True,
            bootstyle="dark"
        )
        self.log_text.pack(fill='both', expand=True)

        # 设置日志文本样式
        self.log_text.text.configure(
            font=('Consolas', 10),  # 稍大的字体
            bg='#1e272e',
            fg='#dfe6e9',
            insertbackground='#dfe6e9',
            selectbackground='#74b9ff',
            padx=10,
            pady=10
        )

        # 配置日志标签颜色
        self.log_text.text.tag_configure('info', foreground='#74b9ff')
        self.log_text.text.tag_configure('success', foreground='#00b894')
        self.log_text.text.tag_configure('warning', foreground='#fdcb6e')
        self.log_text.text.tag_configure('error', foreground='#e17055')
        self.log_text.text.tag_configure('time', foreground='#636e72')

    def _build_status_bar(self, parent):
        """构建底部状态栏"""
        status_frame = ttk.Frame(parent, padding=(0, 3))
        status_frame.pack(fill='x')

        self.status_label = ttk.Label(
            status_frame,
            text="✨ 就绪",
            font=('Segoe UI', 9),
            foreground='#b2bec3'
        )
        self.status_label.pack(side='left')

        # 版本信息
        version_label = ttk.Label(
            status_frame,
            text="v2.0.0",
            font=('Segoe UI', 8),
            foreground='#636e72'
        )
        version_label.pack(side='right')

    # ==================== 日志方法 ====================

    def _log(self, message: str, level: str = 'info'):
        """添加日志消息"""
        timestamp = datetime.now().strftime('%H:%M:%S')

        # 插入时间戳
        self.log_text.text.insert('end', f"[{timestamp}] ", 'time')

        # 级别前缀
        level_prefixes = {
            'info': 'ℹ️ ',
            'success': '✅ ',
            'warning': '⚠️ ',
            'error': '❌ '
        }
        prefix = level_prefixes.get(level, 'ℹ️ ')

        # 插入消息
        self.log_text.text.insert('end', f"{prefix}{message}\n", level)
        self.log_text.text.see('end')
        self.root.update()

    def _set_running_state(self, is_running: bool):
        """设置运行状态"""
        self.running = is_running

        if is_running:
            self.run_btn.configure(state='disabled', text="⏳ 执行中...")
            self.pause_btn.configure(state='normal')
            self.status_label.configure(
                text="🔄 正在执行中...",
                foreground='#00b894'
            )
            self.progress_bar.configure(value=0)
            self.current_progress = 0
            self.progress_label.configure(text="🔄 进度: 0%")
        else:
            self.run_btn.configure(state='normal', text="🚀 开始执行")
            self.pause_btn.configure(state='disabled', text="⏸ 暂停")
            self.status_label.configure(
                text="✨ 就绪",
                foreground='#b2bec3'
            )
            self.progress_bar.configure(value=0)
            self.progress_label.configure(text="⏳ 准备就绪")
            self.paused = False
            self.current_progress = 0

    def _update_progress(self, current: int, total: int = None):
        """更新进度条"""
        if total is not None:
            self.total_count = total
        
        if self.total_count > 0:
            self.current_progress = current
            percent = int((current / self.total_count) * 100)
            self.progress_bar.configure(value=percent)
            self.progress_label.configure(text=f"🔄 进度: {current}/{self.total_count} ({percent}%)")
            self.root.update()

    def _toggle_pause(self):
        """暂停/继续"""
        self.paused = not self.paused

        if self.paused:
            self.pause_btn.configure(text="▶ 继续", bootstyle="success")
            self.status_label.configure(text="⏸ 已暂停", foreground='#fdcb6e')
            # 保持当前进度显示
            if self.total_count > 0:
                percent = int((self.current_progress / self.total_count) * 100)
                self.progress_label.configure(text=f"⏸ 已暂停 ({self.current_progress}/{self.total_count})")
            else:
                self.progress_label.configure(text="⏸ 已暂停")
            self._log("已暂停", 'warning')
            
            # 通知处理器暂停
            if self.processor and hasattr(self.processor, 'set_running'):
                self.processor.set_running(False)
            if self.domain_login and hasattr(self.domain_login, 'set_status'):
                self.domain_login.set_status(False)
        else:
            self.pause_btn.configure(text="⏸ 暂停", bootstyle="warning-outline")
            self.status_label.configure(text="🔄 正在执行中...", foreground='#00b894')
            if self.total_count > 0:
                percent = int((self.current_progress / self.total_count) * 100)
                self.progress_label.configure(text=f"🔄 进度: {self.current_progress}/{self.total_count} ({percent}%)")
            else:
                self.progress_label.configure(text="🔄 正在处理...")
            self._log("已继续", 'info')
            
            # 通知处理器继续
            if self.processor and hasattr(self.processor, 'set_running'):
                self.processor.set_running(True)
            if self.domain_login and hasattr(self.domain_login, 'set_status'):
                self.domain_login.set_status(True)

    # ==================== 业务逻辑 ====================

    def _select_file(self):
        """选择数据文件"""
        path = filedialog.askopenfilename(
            filetypes=[("Excel文件", "*.xlsx *.xls")],
            title="选择数据文件"
        )
        if path:
            self.file_entry.delete(0, ttk.END)
            self.file_entry.insert(0, path)
            self._log(f"已选择文件: {os.path.basename(path)}", 'info')
            
            # 读取 Excel 文件并显示数据条数
            try:
                df = pd.read_excel(path)
                row_count = len(df)
                col_count = len(df.columns)
                self.total_count = row_count  # 保存总条数
                self._log(f"数据加载完成: 共 {row_count} 条记录, {col_count} 个字段", 'success')
                self.progress_label.configure(text=f"已加载 {row_count} 条数据")
            except Exception as e:
                self._log(f"读取文件失败: {str(e)}", 'error')

    def _validate_input(self) -> bool:
        """验证输入有效性"""
        if not self.file_entry.get():
            messagebox.showerror("❌ 输入错误", "请选择数据文件！")
            return False
        if not self.action_var.get():
            messagebox.showerror("❌ 输入错误", "请选择要执行的功能！")
            return False
        return True

    def _start_process(self):
        """启动处理线程"""
        if self.running:
            messagebox.showwarning("⚠️ 警告", "已有任务正在运行！")
            return

        if not self._validate_input():
            return

        self._set_running_state(True)
        self._log("启动处理任务...", 'info')

        threading.Thread(target=self._run_process, daemon=True).start()

    def _run_process(self):
        """后台处理流程"""
        try:
            # 初始化浏览器
            self._log("初始化浏览器引擎...", 'info')
            self._init_browser()

            # 获取参数
            country = self.country_var.get()
            action = self.action_var.get()
            file_name = self.file_entry.get()

            # 获取显示名称
            country_names = {c: n for n, c in self.COUNTRY_OPTIONS}
            action_names = {c: n for n, c in self.ACTION_OPTIONS}

            # 检查暂停状态
            while self.paused:
                time.sleep(0.5)
                if not self.running:
                    return

            # 特殊参数处理
            if action == "payment":
                self._get_payee_code()
                if not self.payee_code:
                    return

            # 创建处理器实例
            self.processor = self._create_processor(country, action)

            # 切换域名
            self._log(f"登录 {country_names.get(country, country)} 站点...", 'info')
            self.domain_login = domain.LoginDomain(self.logger, self.page, country)
            self.domain_login.set_status(True)
            self.domain_login.login_check()

            # 检查暂停状态
            while self.paused:
                time.sleep(0.5)
                if not self.running:
                    return

            # 执行核心操作
            self._log(f"开始执行 {action_names.get(action, action)} 操作...", 'info')
            
            if hasattr(self.processor, 'set_running'):
                self.processor.set_running(True)
            
            self.processor.run(file_name)

            # 检查暂停状态
            while self.paused:
                time.sleep(0.5)
                if not self.running:
                    return

            # 推送数据
            if self.push_msg_var.get() == "1":
                self._log("推送数据到消息队列...", 'info')
                self.processor.push_data_queue()

            self._log("操作成功完成！", 'success')

            # 设置进度为100%
            if self.total_count > 0:
                self._update_progress(self.total_count, self.total_count)
            self.progress_bar.configure(value=100)
            self.progress_label.configure(text="✅ 已完成 100%")

            # 保存日志
            self._save_log()

            # 更新状态栏
            self.root.after(0, lambda: self.status_label.configure(
                text="✅ 任务完成",
                foreground='#00b894'
            ))

        except Exception as e:
            self._log(f"发生错误: {str(e)}", 'error')
            self._log(traceback.format_exc(), 'error')
            self.root.after(0, lambda: self.status_label.configure(
                text="❌ 任务失败",
                foreground='#e17055'
            ))
        finally:
            self._cleanup_resources()

    def _init_browser(self):
        """初始化浏览器配置"""
        self.page = ChromiumPage()
        self.page.set.load_mode.normal()
        self.page.set.when_download_file_exists('overwrite')
        download_path = os.path.join(os.getcwd())
        self.page.set.download_path(download_path)
        self._log(f"下载目录: {download_path}", 'info')

    def _create_processor(self, country: str, action: str):
        """创建功能处理器"""
        # 根据国家编码获取店铺代码
        shop_code_map = {
            "JP": "JP-VC",
            "UK": "UK-VC",
            "DE": "DE-VC",
            "FR": "FR-VC",
            "CA": "CA-VC"
        }
        shop_code = shop_code_map.get(country, "VECELO")

        # 创建处理器
        processors = {
            "payment": lambda: Payment(self.logger, self.page, country, self.payee_code, shop_code),
            "payment_erp": lambda: PaymentPush(self.logger, country, shop_code),
            "return": lambda: ReturnGoods(self.logger, self.page, country, shop_code),
            "spa": lambda: Spa(self.logger, self.page, country, shop_code),
        }

        processor_factory = processors.get(action)
        if processor_factory:
            return processor_factory()
        else:
            raise ValueError(f"未知的功能类型: {action}")

    def _get_payee_code(self):
        """获取回款Code"""
        self.root.after(0, self._show_payee_dialog)
        while self.running and not self.payee_code:
            time.sleep(0.1)

    def _show_payee_dialog(self):
        """显示回款Code输入对话框"""
        self.payee_code = simpledialog.askstring(
            "💰 回款信息",
            "请输入回款Code（从URL参数获取）:",
            parent=self.root,
            initialvalue="VECET"
        )
        if not self.payee_code:
            self._log("用户取消回款Code输入", 'warning')
            self.running = False

    def _save_log(self):
        """保存日志"""
        content = self.log_text.text.get(1.0, ttk.END).strip()

        if content:
            current_dir = os.getcwd()
            current_datetime = datetime.now().strftime('%Y-%m-%d-%H-%M')
            file_name = "running.log"
            result_file_name = f"{current_datetime}_{self.action_var.get()}_{file_name}"
            file_path = os.path.join(current_dir, result_file_name)

            with open(file_path, "w", encoding="utf-8") as file:
                file.write(content)

            self._log(f"日志已保存: {result_file_name}", 'success')

    def _cleanup_resources(self):
        """清理资源"""
        if self.page:
            try:
                self.page.close()
                self._log("浏览器资源已释放", 'info')
            except Exception as e:
                self._log(f"释放浏览器资源时出错: {str(e)}", 'warning')
            finally:
                self.page = None

        self.processor = None
        self.domain_login = None
        self.running = False
        self.root.after(0, lambda: self._set_running_state(False))

    def _process_log_queue(self):
        """处理日志队列"""
        import re
        
        while not self.log_queue.empty():
            msg = self.log_queue.get()

            # 解析进度信息 - 匹配 'index': X 的格式
            index_match = re.search(r"['\"]index['\"]:\s*(\d+)", msg)
            if index_match and self.total_count > 0:
                current_index = int(index_match.group(1))
                self._update_progress(current_index, self.total_count)

            # 判断日志级别
            level = 'info'
            if '错误' in msg or 'Error' in msg or '失败' in msg:
                level = 'error'
            elif '成功' in msg or '完成' in msg:
                level = 'success'
            elif '警告' in msg or '暂停' in msg:
                level = 'warning'

            self._log(msg, level)

        self.root.after(100, self._process_log_queue)

    def run(self):
        """运行应用程序"""
        try:
            self._log("应用程序已启动", 'info')
            self.root.mainloop()
        finally:
            self._cleanup_resources()


if __name__ == "__main__":
    try:
        load_dotenv()
        log = GuiLog()
        app = VCManagerGUI(log)
        app.run()
    except KeyboardInterrupt:
        exit(1)
