# coding: utf-8
"""
Amazon 广告费采集工具
"""

import os
import threading
import queue
import traceback
from datetime import datetime
from DrissionPage import ChromiumPage
import ttkbootstrap as ttk
from ttkbootstrap.scrolled import ScrolledText
from dotenv import load_dotenv

from app.helper import domain, file
from app.logger.gui import GuiLog
from app.vc.advert_cost import AdvertCost
from app.vc.product_sales import ProductSales


class VCManagerGUI:

    # 窗口配置
    WINDOW_WIDTH = 580
    WINDOW_HEIGHT = 720

    # 国家配置
    COUNTRY_OPTIONS = [
        ("全部", "all"),
        ("美国 (US)", "US"),
        ("日本 (JP)", "JP"),
        ("英国 (UK)", "UK"),
        ("法国 (FR)", "FR"),
        ("德国 (DE)", "DE"),
        ("加拿大 (CA)", "CA"),
    ]

    # 功能配置
    ACTION_OPTIONS = [
        ("全部功能", "all"),
        ("广告费用", "advert_cost"),
        ("商品销量", "product_sales"),
    ]

    def __init__(self, logger):
        """初始化应用程序"""
        # 状态变量
        self.domain_login = None
        self.processor = None
        self.page = None
        self.running = False
        self.paused = False
        self.log_queue = queue.Queue()
        self.payee_code = None

        # GUI 组件引用
        self.log_text = None
        self.country_var = None
        self.action_var = None
        self.progress_bar = None
        self.progress_label = None
        self.status_label = None
        self.start_btn = None
        self.pause_btn = None
        self.stats_labels = {}

        # 设置日志
        logger.set_console(self.log_queue)
        self.logger = logger

        # 初始化主窗口 - 使用深色主题
        self.root = ttk.Window(themename="darkly")
        self.root.title("🚀 VC Manager Pro")
        self.root.resizable(False, False)

        # 设置自定义样式
        self._setup_custom_styles()

        # 构建 GUI
        self._build_gui()

        # 窗口居中
        self._center_window()

        # 启动日志处理
        self.root.after(100, self._process_log_queue)

    @staticmethod
    def _setup_custom_styles():
        """设置自定义样式"""
        style = ttk.Style()

        # 标题样式
        style.configure(
            'Title.TLabel',
            font=('Segoe UI', 24, 'bold'),
            foreground='#ffffff'
        )

        # 副标题样式
        style.configure(
            'Subtitle.TLabel',
            font=('Segoe UI', 10),
            foreground='#b2bec3'
        )

        # 区域标题样式
        style.configure(
            'SectionTitle.TLabel',
            font=('Segoe UI', 11, 'bold'),
            foreground='#74b9ff'
        )

        # 统计数字样式
        style.configure(
            'Stats.TLabel',
            font=('Segoe UI', 14, 'bold'),
            foreground='#00cec9'
        )

    def _center_window(self):
        """窗口居中"""
        screen_width = self.root.winfo_screenwidth()
        screen_height = self.root.winfo_screenheight()

        x = int((screen_width - self.WINDOW_WIDTH) / 2)
        y = int((screen_height - self.WINDOW_HEIGHT) / 3)

        self.root.geometry(f"{self.WINDOW_WIDTH}x{self.WINDOW_HEIGHT}+{x}+{y}")

    def _build_gui(self):
        """构建完整 GUI 界面"""
        # 主容器
        main_container = ttk.Frame(self.root, padding=15)
        main_container.pack(fill='both', expand=True)

        # 1. 标题区域
        self._build_header(main_container)

        # 2. 配置卡片
        self._build_config_card(main_container)

        # 3. 统计卡片
        self._build_stats_card(main_container)

        # 4. 日志区域
        self._build_log_area(main_container)

        # 5. 底部状态栏
        self._build_status_bar(main_container)

    @staticmethod
    def _build_header(parent):
        """构建标题区域"""
        header_frame = ttk.Frame(parent)
        header_frame.pack(fill='x', pady=(0, 10))

        # 主标题
        title_label = ttk.Label(
            header_frame,
            text="🚀 VC Manager",
            style='Title.TLabel'
        )
        title_label.pack()

        # 副标题
        subtitle_label = ttk.Label(
            header_frame,
            text="Amazon Vendor Central 数据采集工具，支持多站点批量处理",
            style='Subtitle.TLabel'
        )
        subtitle_label.pack(pady=(3, 0))

    def _build_config_card(self, parent):
        """构建配置卡片区域"""
        # 卡片容器
        card_frame = ttk.LabelFrame(
            parent,
            text=" ⚙️ 采集配置 ",
            padding=10,
            style="info"
        )
        card_frame.pack(fill='x', pady=(0, 10))

        # === 国家选择区 ===
        self.country_var = ttk.StringVar(value="all")

        # 使用 grid 布局让按钮整齐排列
        country_grid = ttk.Frame(card_frame)
        country_grid.pack(fill='x', pady=(0, 5))

        for i, (name, code) in enumerate(self.COUNTRY_OPTIONS):
            row = i // 4
            col = i % 4
            btn = ttk.Radiobutton(
                country_grid,
                text=name,
                value=code,
                variable=self.country_var,
                style="info-toolbutton",
                width=10
            )
            btn.grid(row=row, column=col, padx=2, pady=2, sticky='ew')

        # 让列等宽
        for col in range(4):
            country_grid.columnconfigure(col, weight=1)

        # === 功能选择区 ===
        self.action_var = ttk.StringVar(value="all")

        action_grid = ttk.Frame(card_frame)
        action_grid.pack(fill='x', pady=(0, 5))

        for i, (name, code) in enumerate(self.ACTION_OPTIONS):
            btn = ttk.Radiobutton(
                action_grid,
                text=name,
                value=code,
                variable=self.action_var,
                style="success-toolbutton",
                width=12
            )
            btn.grid(row=0, column=i, padx=2, pady=2, sticky='ew')

        # 让列等宽
        for col in range(len(self.ACTION_OPTIONS)):
            action_grid.columnconfigure(col, weight=1)

        # 分隔线
        ttk.Separator(card_frame, style="secondary").pack(fill='x', pady=10)

        # === 进度区域 ===
        progress_frame = ttk.Frame(card_frame)
        progress_frame.pack(fill='x')

        self.progress_label = ttk.Label(
            progress_frame,
            text="准备就绪",
            font=('Segoe UI', 9),
            foreground='#b2bec3'
        )
        self.progress_label.pack(anchor='w')

        self.progress_bar = ttk.Progressbar(
            progress_frame,
            mode='indeterminate',
            style="success-striped",
            length=300
        )
        self.progress_bar.pack(fill='x', pady=(5, 10))

        # === 操作按钮 ===
        button_frame = ttk.Frame(card_frame)
        button_frame.pack()

        self.start_btn = ttk.Button(
            button_frame,
            text="🚀 开始采集",
            command=self._start_process,
            style="success",
            width=14
        )
        self.start_btn.pack(side='left', padx=5)

        self.pause_btn = ttk.Button(
            button_frame,
            text="⏸ 暂停",
            command=self._toggle_pause,
            style="warning-outline",
            width=14,
            state='disabled'
        )
        self.pause_btn.pack(side='left', padx=5)

    def _build_stats_card(self, parent):
        """构建统计信息卡片"""
        stats_frame = ttk.Frame(parent)
        stats_frame.pack(fill='x', pady=(0, 5))

        # 三列统计 - 紧凑布局
        stats_config = [
            ("countries", "处理国家", "0", "info"),
            ("actions", "执行功能", "0", "success"),
            ("errors", "错误数量", "0", "danger"),
        ]

        for stat_id, label_text, value, sty in stats_config:
            stat_card = ttk.LabelFrame(
                stats_frame,
                text=f" {label_text} ",
                padding=(10, 2),
                style=sty
            )
            stat_card.pack(side='left', fill='x', expand=True, padx=3)

            value_label = ttk.Label(
                stat_card,
                text=value,
                style='Stats.TLabel'
            )
            value_label.pack()

            self.stats_labels[stat_id] = value_label

    def _build_log_area(self, parent):
        """构建日志显示区域"""
        log_frame = ttk.LabelFrame(
            parent,
            text=" 运行日志 ",
            padding=8,
            style="secondary"
        )
        log_frame.pack(fill='both', expand=True, pady=(0, 5))

        # 使用 ScrolledText
        self.log_text = ScrolledText(
            log_frame,
            height=18,
            autohide=True,
            bootstyle="dark"
        )
        self.log_text.pack(fill='both', expand=True)

        # 设置日志文本样式
        self.log_text.text.configure(
            font=('Consolas', 9),
            bg='#1e272e',
            fg='#dfe6e9',
            insertbackground='#dfe6e9',
            selectbackground='#74b9ff',
            padx=10,
            pady=10
        )

        # 配置日志标签颜色
        self.log_text.text.tag_configure('info', foreground='#74b9ff')
        self.log_text.text.tag_configure('success', foreground='#00b894')
        self.log_text.text.tag_configure('warning', foreground='#fdcb6e')
        self.log_text.text.tag_configure('error', foreground='#e17055')
        self.log_text.text.tag_configure('time', foreground='#636e72')

    def _build_status_bar(self, parent):
        """构建底部状态栏"""
        status_frame = ttk.Frame(parent, padding=(0, 5))
        status_frame.pack(fill='x')

        self.status_label = ttk.Label(
            status_frame,
            text="就绪",
            font=('Segoe UI', 9),
            foreground='#b2bec3'
        )
        self.status_label.pack(side='left')

        # 版本信息
        version_label = ttk.Label(
            status_frame,
            text="v2.0.0",
            font=('Segoe UI', 8),
            foreground='#636e72'
        )
        version_label.pack(side='right')

    # ==================== 日志方法 ====================

    def _log(self, message: str, level: str = 'info'):
        """添加日志消息"""
        timestamp = datetime.now().strftime('%H:%M:%S')

        # 插入时间戳
        self.log_text.text.insert('end', f"[{timestamp}] ", 'time')

        # 级别前缀
        level_prefixes = {
            'info': 'i',
            'success': '+',
            'warning': '!',
            'error': 'x'
        }
        prefix = level_prefixes.get(level, 'i')

        # 插入消息
        self.log_text.text.insert('end', f"{prefix}  {message}\n", level)
        self.log_text.text.see('end')
        self.root.update()

    def _update_stats(self, countries: int = None, actions: int = None, errors: int = None):
        """更新统计数据"""
        if countries is not None:
            self.stats_labels['countries'].configure(text=str(countries))
        if actions is not None:
            self.stats_labels['actions'].configure(text=str(actions))
        if errors is not None:
            self.stats_labels['errors'].configure(text=str(errors))

    def _set_running_state(self, is_running: bool):
        """设置运行状态"""
        self.running = is_running

        if is_running:
            self.start_btn.configure(state='disabled', text="⏳ 执行中...")
            self.pause_btn.configure(state='normal')
            self.status_label.configure(
                text="🔄 正在采集中...",
                foreground='#00b894'
            )
            self.progress_bar.start(15)
        else:
            self.start_btn.configure(state='normal', text="🚀 开始采集")
            self.pause_btn.configure(state='disabled', text="⏸ 暂停")
            self.status_label.configure(
                text="就绪",
                foreground='#b2bec3'
            )
            self.progress_bar.stop()

    # ==================== 业务逻辑 ====================

    def _get_selected_country(self):
        """获取选中的国家代码"""
        return self.country_var.get()

    def _get_selected_action(self):
        """获取选中的功能代码"""
        return self.action_var.get()

    def _start_process(self):
        """开始处理"""
        if self.running:
            self._log("任务已在运行中", 'warning')
            return

        self._set_running_state(True)
        self.paused = False
        self._update_stats(countries=0, actions=0, errors=0)
        self._log("启动采集任务...", 'info')

        threading.Thread(target=self._run_process, daemon=True).start()

    def _toggle_pause(self):
        """暂停/继续"""
        self.paused = not self.paused

        if self.paused:
            self.pause_btn.configure(text="▶ 继续", bootstyle="success")
            self.progress_bar.stop()
            self.status_label.configure(text="⏸ 已暂停", foreground='#fdcb6e')
            self._log("已暂停", 'warning')
            if self.processor:
                self.processor.set_running(False)
            if self.domain_login:
                self.domain_login.set_status(False)
        else:
            self.pause_btn.configure(text="⏸ 暂停", bootstyle="warning-outline")
            self.progress_bar.start(15)
            self.status_label.configure(text="🔄 正在采集中...", foreground='#00b894')
            self._log("已继续", 'info')
            if self.processor:
                self.processor.set_running(True)
            if self.domain_login:
                self.domain_login.set_status(True)

    def _run_process(self):
        """后台处理流程"""
        import time
        error_count = 0
        country_count = 0
        action_count = 0

        try:
            self._log("初始化浏览器引擎...", 'info')
            self._init_browser()

            country_code = self._get_selected_country()
            action_code = self._get_selected_action()

            # 处理国家选择
            if country_code == "all":
                countries = [(name, code) for name, code in self.COUNTRY_OPTIONS if code != "all"]
            else:
                countries = [(None, country_code)]

            # 登录所有选择的国家
            for name, code in countries:
                # 检查暂停状态
                while self.paused:
                    time.sleep(0.5)
                    if not self.running:
                        return

                if not self.running:
                    self._log("任务已停止", 'warning')
                    return

                display_name = name.split(" ")[0] if name else code
                self._log(f"登录 {display_name}...", 'info')
                self.domain_login = domain.LoginDomain(self.logger, self.page, code)
                self.domain_login.set_status(True)
                self.domain_login.login_check()

            # 执行任务
            for name, code in countries:
                # 检查暂停状态
                while self.paused:
                    time.sleep(0.5)
                    if not self.running:
                        return

                if not self.running:
                    self._log("任务已停止", 'warning')
                    return

                country_count += 1
                self._update_stats(countries=country_count)

                display_name = name.split(" ")[0] if name else code

                # 处理动作选择
                if action_code == "all":
                    actions = [(n, c) for n, c in self.ACTION_OPTIONS if c != "all"]
                else:
                    actions = [(None, action_code)]

                for action_name, action_val in actions:
                    # 检查暂停状态
                    while self.paused:
                        time.sleep(0.5)
                        if not self.running:
                            return

                    if not self.running:
                        self._log("任务已停止", 'warning')
                        return

                    action_count += 1
                    self._update_stats(actions=action_count)

                    display_action = action_name or action_val
                    self._log(f"[{display_name}] 执行 {display_action}...", 'info')

                    try:
                        self.processor = self._get_processor(action_val, code)
                        if self.processor:
                            self.processor.set_running(True)
                            self.processor.run("")
                            self.processor.push_data_queue()
                            self._log(f"[{display_name}] {display_action} 完成", 'success')
                    except Exception as e:
                        error_count += 1
                        self._update_stats(errors=error_count)
                        self._log(f"[{display_name}] {display_action} 失败: {e}", 'error')

            self._log("全部任务执行完成！", 'success')
            self.root.after(0, lambda: self.status_label.configure(
                text="✅ 任务完成",
                foreground='#00b894'
            ))

        except Exception as e:
            error_count += 1
            self._update_stats(errors=error_count)
            self._log(f"发生错误: {e}", 'error')
            self._log(traceback.format_exc(), 'error')
            self.root.after(0, lambda: self.status_label.configure(
                text="❌ 任务失败",
                foreground='#e17055'
            ))
        finally:
            self._cleanup_resources()

    def _get_processor(self, action, country):
        """获取处理器实例"""
        processors = {
            "advert_cost": AdvertCost,
            "product_sales": ProductSales
        }
        processor_class = processors.get(action)
        if processor_class:
            return processor_class(self.logger, self.page, str(country))
        return None

    def _init_browser(self):
        """初始化浏览器"""
        self.page = ChromiumPage()
        self.page.set.load_mode.normal()
        self.page.set.when_download_file_exists('overwrite')
        self.page.set.download_path(os.getcwd())
        file.make_dir('products')
        self._log("浏览器初始化完成", 'success')

    def _cleanup_resources(self):
        """清理资源"""
        if self.page:
            try:
                self.page.close()
                self._log("浏览器已关闭", 'info')
            except Exception as e:
                self._log(f"关闭浏览器出错: {e}", 'warning')
            finally:
                self.page = None

        self.running = False
        self.paused = False
        self.root.after(0, lambda: self._set_running_state(False))

    def _process_log_queue(self):
        """处理日志队列"""
        while not self.log_queue.empty():
            msg = self.log_queue.get()

            # 判断日志级别
            level = 'info'
            if '错误' in msg or 'Error' in msg or '失败' in msg:
                level = 'error'
            elif '成功' in msg or '完成' in msg:
                level = 'success'
            elif '警告' in msg or '暂停' in msg:
                level = 'warning'

            self._log(msg, level)

        self.root.after(150, self._process_log_queue)

    def run(self):
        """运行应用程序"""
        try:
            self._log("应用程序已启动", 'info')
            self.root.mainloop()
        finally:
            self._cleanup_resources()


if __name__ == "__main__":
    try:
        load_dotenv()
        log = GuiLog()
        app = VCManagerGUI(log)
        app.run()
    except KeyboardInterrupt:
        exit(1)
