# coding: utf-8

import os
import threading
import queue
import time
import traceback
from tkinter import messagebox
from DrissionPage import ChromiumPage
import ttkbootstrap as ttk
from ttkbootstrap.constants import *

from app.helper import domain,file
from app.logger.gui import GuiLog
from app.vc.advert_cost import AdvertCost
from app.vc.product_sales import ProductSales


class VCManagerGUI(ttk.Window):
    def __init__(self, logger):
        super().__init__(themename="cosmo")

        self.run_btn = None
        self.shop_entry = None
        self.country_var = None
        self.action_var = None
        self.log_text = None
        self.title("广告费工具")
        self.geometry("520x600")

        # 设置窗口属性禁止缩放
        self.resizable(False, False)

        # 初始化状态变量
        self.page = None
        self.running = False
        self.log_queue = queue.Queue()
        self.payee_code = None

        # 设置日志处理
        logger.set_console(self.log_queue)
        self.logger = logger

        # 设置窗口尺寸并居中
        self._center_window()

        # 创建界面组件
        self.create_widgets()

        # 启动日志处理循环
        self.after(100, self.process_log_queue)

    def _center_window(self):
        """设置窗口居中"""
        window_width = 520
        window_height = 600

        # 获取屏幕尺寸
        screen_width = self.winfo_screenwidth()
        screen_height = self.winfo_screenheight()

        # 计算居中坐标
        x = (screen_width - window_width) // 2
        y = (screen_height - window_height) // 2

        # 设置窗口位置
        self.geometry(f"{window_width}x{window_height}+{x}+{y}")

    def create_widgets(self):
        """创建主界面布局"""
        main_frame = ttk.Frame(self)
        main_frame.pack(fill=BOTH, expand=True, padx=15, pady=15)

        # 配置区域
        config_frame = ttk.Labelframe(main_frame, text="配置参数", padding=10)
        config_frame.pack(fill=X, pady=10)

        # 国家选择
        country_frame = ttk.Frame(config_frame)
        country_frame.grid(row=0, column=0, sticky=W, pady=5)
        ttk.Label(country_frame, text="国家:", width=6).pack(side=LEFT)
        self.country_var = ttk.StringVar(value="US")
        countries = [
            ("美国", "US"),
            # ("英国", "UK"),
            ("日本", "JP"),
            # ("法国", "FR"),
            # ("德国", "DE"),
            # ("加拿大", "CA"),
        ]
        for i, (text, value) in enumerate(countries):
            rb = ttk.Radiobutton(country_frame,text=text,variable=self.country_var,value=value)
            rb.pack(side=LEFT, padx=(10 if i != 0 else 0))

        # 功能选择
        action_frame = ttk.Frame(config_frame)
        action_frame.grid(row=1, column=0, sticky=W, pady=5)
        ttk.Label(action_frame, text="功能:", width=6).pack(side=LEFT)
        self.action_var = ttk.StringVar()
        actions = [
            ("广告费", "advert_cost"),
            ("商品销量", "product_sales")
        ]
        for i, (text, value) in enumerate(actions):
            rb = ttk.Radiobutton(action_frame, text=text, variable=self.action_var, value=value)
            rb.pack(side=LEFT, padx=(10 if i != 0 else 0))

        # 控制按钮
        btn_frame = ttk.Frame(main_frame)
        btn_frame.pack(fill=X, pady=15)
        self.run_btn = ttk.Button(btn_frame, text="开始执行", command=self.start_process,
                                  style=PRIMARY, width=12)
        self.run_btn.pack(side=RIGHT, padx=5)

        # 日志显示
        log_frame = ttk.Labelframe(main_frame, text="操作日志", padding=10)
        log_frame.pack(fill=BOTH, expand=True)

        self.log_text = ttk.ScrolledText(log_frame, state=DISABLED)
        self.log_text.pack(fill=BOTH, expand=True)

    def start_process(self):
        """启动处理线程"""
        if self.running:
            messagebox.showwarning("警告", "已有任务正在运行")
            return

        if not self.action_var.get():
            messagebox.showerror("输入错误", "请选择要执行的功能")
            return

        self.running = True
        self.run_btn.config(state=ttk.DISABLED)
        threading.Thread(target=self.run_process, daemon=True).start()

    def run_process(self):
        """后台处理流程"""
        try:
            # 初始化浏览器
            self.logger.info("初始化浏览器引擎...")
            self.init_browser()
            # 获取参数
            country = self.country_var.get()
            action = self.action_var.get()

            # 切换域名
            domain.domain_page(self.logger, self.page, country)

            # 执行核心操作
            self.logger.info(f"开始执行 {action} 操作...")

            processor = None
            if action == "advert_cost":
                processor = AdvertCost(self.logger, self.page, str(country))
            elif action == "product_sales":
                processor = ProductSales(self.logger, self.page, str(country))

            processor.run("")
            processor.push_data_queue()

            self.logger.info("操作成功完成！")
        except Exception as e:
            self.logger.info(f"发生错误：{str(e)}")
            self.logger.info(traceback.format_exc())
        finally:
            pass
            self.cleanup_resources()

    def init_browser(self):
        """初始化浏览器配置"""
        self.page = ChromiumPage()
        self.page.set.load_mode.normal()
        self.page.set.when_download_file_exists('overwrite')
        download_path = os.path.join(os.getcwd())
        self.page.set.download_path(download_path)
        file.make_dir('products')

    def log(self, message):
        """记录日志到队列"""
        self.logger.info(message)
        timestamp = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
        self.log_queue.put(f"[{timestamp}] {message}")

    def process_log_queue(self):
        """处理日志队列"""
        while not self.log_queue.empty():
            msg = self.log_queue.get()
            self.log_text.configure(state=ttk.NORMAL)
            self.log_text.insert(ttk.END, msg + "\n")
            self.log_text.configure(state=ttk.DISABLED)
            self.log_text.see(ttk.END)
        self.after(100, self.process_log_queue)

    def clear_log(self):
        """清除日志"""
        self.log_text.configure(state=ttk.NORMAL)
        self.log_text.delete(1.0, ttk.END)
        self.log_text.configure(state=ttk.DISABLED)

    def cleanup_resources(self):
        """清理资源"""
        if self.page:
            try:
                self.page.close()
                self.logger.info("浏览器资源已释放")
            except Exception as e:
                self.logger.info(f"释放浏览器资源时出错：{str(e)}")
            finally:
                self.page = None
        self.running = False
        self.after(0, lambda: self.run_btn.config(state=ttk.NORMAL))


if __name__ == "__main__":
    try:
        log = GuiLog()
        app = VCManagerGUI(log)
        app.mainloop()
    except KeyboardInterrupt:
        exit(1)
